from contextlib import closing
from kenji import Graph, V
from sqlite3 import ProgrammingError
from threading import Thread


def test_concurrency(graph):
    def store(x):
        def function():
            graph.store(V(1).knows(x))
        return function

    stored = (5, 6, 7, 8)
    threads = [Thread(target=store(i)) for i in stored]
    [thread.start() for thread in threads]
    [thread.join() for thread in threads]

    for i in stored:
        assert graph.exists(V(1).knows(i))


def test_store():
    graph = Graph(graphs=['knows'])
    graph.store(V(1).knows(10))
    assert graph.exists(V(1).knows(10))


def test_unsafe():
    graph = Graph(unsafe=True)
    with closing(graph.db.cursor()) as cursor:
        cursor.execute('PRAGMA synchronous')
        assert not next(cursor)[0]


def test_delete(graph):
    graph.delete(V().knows(2))
    assert 2 not in list(graph.select(V(1).knows))

    graph.delete(V(1).knows)
    assert not list(graph.select(V(1).knows))

    graph.store(V(1).knows(2))
    graph.delete(V().knows)
    assert not list(graph.select(V(1).knows))


def test_close(graph):
    graph.close()
    try:
        graph.db.execute('SELECT * FROM knows;')
        raise AssertionError('connection is not closed!')
    except ProgrammingError:
        pass


def test_graph_relation(graph):
    assert graph.relation_between('person', V(1) >= V(2)) == ['knows', 'likes']
    assert graph.relation_between('person', V(0) >= V(2)) == []


def test_graph_exists(graph):
    assert graph.exists(V(1).knows(2))
    assert not graph.exists(V(1).knows(10))
