"""
Implements the main transaction object.
"""

from contextlib import closing
from kenji.sql import write_relation, delete_relation

__all__ = ['Transaction']


class AbortSignal(Exception):
    """
    A "signal" raised when the :meth:``Transaction.abort``
    method is called.
    """
    pass


class Transaction(object):
    """
    A transaction represents an atomic set of operations
    that are either all performed in the database or none
    are performed. Note that transactions are `not`
    thread safe.
    """

    def __init__(self, db):
        """
        Create a new transaction object.

        :param db: An SQLite connection.
        """
        self.db = db
        self.ops = []

    @property
    def defined(self):
        """
        Property dictating of there are any operations
        (pending changes) to be done on the database.
        """
        return bool(self.ops)

    def abort(self):
        """
        Aborts the transaction and raises an
        :class:``AbortSignal``, which is not propogated
        to the block of code outside the ``with`` block
        and silenced.
        """
        self.ops = []
        raise AbortSignal()

    def store(self, edge):
        """
        Store an edge.

        :param edge: The edge to store.
        """
        self.ops.append((write_relation, edge))

    def delete(self, edge):
        """
        Delete possibly multiple edges.

        :param edge: The edge to delete.
        """
        self.ops.append((delete_relation, edge))

    def commit(self):
        """
        Commits the transaction. Not to be called by
        application code since this is called automagically
        by context managers.
        """
        cursor = self.db.cursor()
        with self.db:
            cursor.execute('begin')
            for ins, edge in self.ops:
                cursor.execute(*ins(
                    edge.src,
                    edge.relation,
                    edge.dst
                ))
