#
# compare dive profiles on one plot
#

library(Hmisc)
library(grid)
library(colorspace)

if (length(kz.args) < 2) {
    stop('Arguments required: output file, output file format')
}

kz.args.fout = kz.args[1]
kz.args.fmt = kz.args[2]
kz.args.sig = '--sig' %in% kz.args

kz.args.width = 10
kz.args.height = 5

if (kz.args.fmt == 'pdf') {
    cairo_pdf(kz.args.fout, width=kz.args.width, height=kz.args.height, onefile=TRUE)
} else if (kz.args.fmt == 'png') {
    fimg = png
    kz.args.width = 900
    kz.args.height = 450
    png(kz.args.fout, width=kz.args.width, height=kz.args.height, res=96)
} else if (kz.args.fmt == 'svg') {
    svg(kz.args.fout, width=kz.args.width, height=kz.args.height)
}

if (is.null(kz.dives.ui$title))
    par(mar=c(5, 4, 1, 2) + 0.1)

# find max times and max depths within dive profiles
lim = aggregate(kz.profiles[c('time', 'depth')], by=kz.profiles['dive'],
    function(a, b) max(a), max(b))

xl = c(0, max(lim$time) / 60.0)
yl = rev(c(0, max(lim$depth)))
plot(NA, xlim=xl, ylim=yl, xlab='Time [min]', ylab='Depth [m]')

# first the grid
minor.tick(nx=5, ny=2)
grid()

n = nrow(kz.dives)
cols = diverge_hcl(n)

# then plot the profiles
for (i in 1:n) {
    dp = kz.profiles[kz.profiles$dive == i, ]
    lines(dp$time / 60.0, dp$depth, col=cols[i])
}

if (!is.null(kz.dives.ui$label)) {
    lscale = ifelse(n > 10, 0.7, 1.0)
    legend('bottomright', as.vector(kz.dives.ui$label), col=cols, lwd=1,
        inset=c(0.02, 0.05), ncol=ceiling(n / 10), cex=lscale)
}

if (kz.args.sig)
    grid.text(sprintf('generated by kenozooid ver. %s', kz.version),
        x=0.99, y=0.01, just=c('right', 'bottom'),
        gp=gpar(cex=0.6, fontface='italic'))

dev.off()

# vim: sw=4:et:ai
