"""
Parser combinators, for combining parsers.
"""

import itertools

from .parser import Parser, Unexpected, State

__all__ = ["many", "many1", "count", "optional", "sequence", "choice",
           "sequence_l", "sequence_r", "between", "sep_by", "sep_by1", "try_",
           "mapf", "mapf_star"]


class many(Parser):
    """
    Run a parser zero or more times. Returns a list of all the parser results.
    """

    def __init__(self, parser):
        self.parser = parser

    def do_parse(self, state):
        xs = []
        while True:
            try:
                xs.append(self.parser.do_parse(state))
            except Unexpected:
                break
        return xs

    def __repr__(self):
        return "many({parser!r})".format(**self.__dict__)


class many1(Parser):
    """
    Run a parser one or more times. Returns a list of all the parser results.
    """

    def __init__(self, parser):
        self.parser = parser

    def do_parse(self, state):
        xs = [self.parser.do_parse(state)]
        while True:
            try:
                xs.append(self.parser.do_parse(state))
            except Unexpected:
                break
        return xs

    def __repr__(self):
        return "many1({parser!r})".format(**self.__dict__)


class count(Parser):
    """
    Run the parser exactly n times. Returns a list of all parser results.
    """

    def __init__(self, n, parser):
        self.n = n
        self.parser = parser

    def do_parse(self, state):
        return [self.parser.do_parse(state) for _ in range(self.n)]

    def __repr__(self):
        return "count({n!r}, {parser!r})".format(**self.__dict__)


class optional(Parser):
    """
    Optionally run a parser. Returns either ``None`` or the parser result.
    """

    def __init__(self, parser, default=lambda: None):
        self.parser = parser
        self.default = default

    def do_parse(self, state):
        try:
            return self.parser.do_parse(state)
        except Unexpected:
            return self.default()

    def __repr__(self):
        return "optional({parser!r}, ...)".format(**self.__dict__)


class sequence(Parser):
    """
    Run parsers in sequence. Returns a list of all the parser results.
    """

    def __init__(self, *parsers):
        self.parsers = parsers

    def do_parse(self, state):
        xs = []

        for parser in self.parsers:
            xs.append(parser.do_parse(state))

        return xs

    def __repr__(self):
        return "sequence{parsers!r}".format(**self.__dict__)

    def __add__(self, rhs):
        return self.__class__(*(self.parsers + (rhs,)))


class sequence_l(Parser):
    """
    Run two parsers in sequence. Returns the result on the left.
    """

    def __init__(self, l, r):
        self.l = l
        self.r = r

    def do_parse(self, state):
        result = self.l.do_parse(state)
        self.r.do_parse(state)
        return result

    def __repr__(self):
        return "sequence_l({l!r}, {r!r})".format(**self.__dict__)


class sequence_r(Parser):
    """
    Run two parsers in sequence. Returns the result on the right.
    """

    def __init__(self, l, r):
        self.l = l
        self.r = r

    def do_parse(self, state):
        self.l.do_parse(state)
        result = self.r.do_parse(state)
        return result

    def __repr__(self):
        return "sequence_r({l!r}, {r!r})".format(**self.__dict__)


class choice(Parser):
    """
    Run any of the parser alternatives.
    """

    def __init__(self, *parsers):
        self.parsers = parsers

    def do_parse(self, state):
        expected = set([])

        for parser in self.parsers:
            try:
                return parser.do_parse(state)
            except Unexpected as e:
                expected.update(e.expected)

        i, x = state.peek()
        state.expect(i, frozenset(expected), x)

    def __repr__(self):
        return "choice{parsers!r}".format(**self.__dict__)

    def __or__(self, rhs):
        return self.__class__(*(self.parsers + (rhs,)))


class between(Parser):
    """
    Run a parser between two other parsers, returning only the middle result.
    Is an optimized version of ``open >> parser << close``.
    """

    def __init__(self, open, close, parser):
        self.open = open
        self.close = close
        self.parser = parser

    def do_parse(self, state):
        self.open.do_parse(state)
        r = self.parser.do_parse(state)
        self.close.do_parse(state)

        return r

    def __repr__(self):
        return "between({open!r}, {close!r}, {parser!r})".format(
            **self.__dict__)


class sep_by(Parser):
    """
    Parse zero or more elements, separated by another parser. Returns a list of
    results.
    """

    def __init__(self, parser, sep):
        self.parser = parser
        self.sep = sep

    def do_parse(self, state):
        xs = []

        try:
            xs.append(self.parser.do_parse(state))
        except Unexpected:
            return xs

        while True:
            try:
                self.sep.do_parse(state)
            except Unexpected:
                break

            xs.append(self.parser.do_parse(state))

        return xs

    def __repr__(self):
        return "sep_by({parser!r}, {sep!r})".format(
            **self.__dict__)


class sep_by1(Parser):
    """
    Parse one or more elements, separated by another parser. Returns a list of
    results.
    """

    def __init__(self, parser, sep):
        self.parser = parser
        self.sep = sep

    def do_parse(self, state):
        xs = [self.parser.do_parse(state)]
        while True:
            try:
                self.sep.do_parse(state)
            except Unexpected:
                break

            xs.append(self.parser.do_parse(state))

        return xs

    def __repr__(self):
        return "sep_by1({parser!r}, {sep!r})".format(
            **self.__dict__)


class try_(Parser):
    """
    Encapsulate a parser with infinite lookahead (rather than LL(1)).
    """

    def __init__(self, parser):
        self.parser = parser

    def do_parse(self, state):
        i, x = state.peek()

        state._it, it2 = itertools.tee(state._it)

        state2 = State(itertools.chain([x], it2))
        state2.index = i - 1

        r = self.parser.do_parse(state2)

        i, x = state2.peek()

        state._has_peek = False
        state._it = itertools.chain([x], state2._it)
        state.index = i - 1

        return r

    def __repr__(self):
        return "try_({parser!r})".format(**self.__dict__)


class _mapfed(Parser):
    def __init__(self, parser, f):
        self.parser = parser
        self.f = f

    def do_parse(self, state):
        return self.f(self.parser.do_parse(state))


def mapf(parser):
    """
    Lift a function into the parser, such that it can be applied to transform
    parser results.
    """

    def _closure(f):
        return _mapfed(parser, f)

    return _closure


class _mapfed_star(Parser):
    def __init__(self, parser, f):
        self.parser = parser
        self.f = f

    def do_parse(self, state):
        return self.f(*self.parser.do_parse(state))


def mapf_star(parser):
    """
    Lift a function into the parser, such that it can be applied to transform
    parser results. This version will expand the parser result as a list of
    arguments to the function.
    """

    def _closure(f):
        return _mapfed_star(parser, f)

    return _closure
