from kessel import parser, combinators


def test_many():
    p = combinators.many(parser.single("a"))
    assert p.parse("aaaaaa") == ["a", "a", "a", "a", "a", "a"]


def test_many_repr():
    assert repr(combinators.many(parser.single("a"))) == \
        "many(single('a'))"


def test_many1():
    p = combinators.many1(parser.single("a"))
    assert p.parse("aaaaaa") == ["a", "a", "a", "a", "a", "a"]


def test_bad_many1():
    p = combinators.many1(parser.single("a"))
    try:
        p.parse("")
    except parser.Unexpected as e:
        if e.expected != frozenset(["a"]) or e.actual != parser.EOF:
            raise
    else:
        assert False, "should have thrown exception"


def test_many1_repr():
    assert repr(combinators.many1(parser.single("a"))) == \
        "many1(single('a'))"


def test_zero_count():
    p = combinators.count(0, parser.single("a"))
    assert p.parse([]) == []


def test_count():
    p = combinators.count(6, parser.single("a"))
    assert p.parse("aaaaaa") == ["a", "a", "a", "a", "a", "a"]


def test_bad_count():
    p = combinators.count(7, parser.single("a"))
    try:
        p.parse("aaaaaa")
    except parser.Unexpected as e:
        if e.expected != frozenset(["a"]) or e.actual != parser.EOF:
            raise
    else:
        assert False, "should have thrown exception"


def test_count_repr():
    assert repr(combinators.count(6, parser.single("a"))) == \
        "count(6, single('a'))"


def test_optional():
    p = combinators.optional(parser.single("a"))
    assert p.parse("a") == "a"


def test_optional_empty():
    p = combinators.optional(parser.single("a"))
    assert p.parse("") is None


def test_optional_repr():
    assert repr(combinators.optional(parser.single("a"))) == \
        "optional(single('a'), ...)"


def test_sequence():
    p = combinators.sequence(parser.single("a"), parser.single("b"))
    assert p.parse("ab") == ["a", "b"]


def test_sequence_operator():
    p = parser.single("a") + parser.single("b")
    assert p.parse("ab") == ["a", "b"]


def test_sequence_operator_3():
    p = parser.single("a") + parser.single("b") + parser.single("c")
    assert p.parse("abc") == ["a", "b", "c"]


def test_sequence_repr():
    assert repr(combinators.sequence(parser.single("a"),
                                     parser.single("b"))) == \
        "sequence(single('a'), single('b'))"


def test_sequence_l():
    p = combinators.sequence_l(parser.single("a"), parser.single("b"))
    assert p.parse("ab") == "a"


def test_sequence_l_operator():
    p = parser.single("a") << parser.single("b")
    assert p.parse("ab") == "a"


def test_sequence_l_repr():
    assert repr(combinators.sequence_l(parser.single("a"),
                                       parser.single("b"))) == \
        "sequence_l(single('a'), single('b'))"


def test_sequence_r():
    p = combinators.sequence_r(parser.single("a"), parser.single("b"))
    assert p.parse("ab") == "b"


def test_sequence_r_operator():
    p = parser.single("a") >> parser.single("b")
    assert p.parse("ab") == "b"


def test_sequence_r_repr():
    assert repr(combinators.sequence_r(parser.single("a"),
                                       parser.single("b"))) == \
        "sequence_r(single('a'), single('b'))"


def test_sequence_partial():
    p = combinators.sequence(parser.single("a"), parser.single("b"))
    try:
        p.parse("aa")
    except parser.Unexpected as e:
        if e.expected != frozenset(["b"]) or e.actual != "a":
            raise
    else:
        assert False, "should have thrown exception"


def test_choice_first():
    p = combinators.choice(parser.single("a"), parser.single("b"))
    assert p.parse("a") == "a"


def test_choice_second():
    p = combinators.choice(parser.single("a"), parser.single("b"))
    assert p.parse("b") == "b"


def test_choice_no_match():
    p = combinators.choice(parser.single("a"), parser.single("b"))
    assert p.parse("b") == "b"


def test_choice_operator():
    p = parser.single("a") | parser.single("b")
    assert p.parse("a") == "a"


def test_choice_operator_3():
    p = parser.single("a") | parser.single("b") | parser.single("c")
    assert p.parse("a") == "a"


def test_choice_repr():
    assert repr(combinators.choice(parser.single("a"),
                                   parser.single("b"))) == \
        "choice(single('a'), single('b'))"


def test_between():
    p = combinators.between(parser.single("("), parser.single(")"),
                            parser.single("a"))
    assert p.parse("(a)") == "a"


def test_sep_by():
    p = combinators.sep_by(parser.single("a"), parser.single(","))
    assert p.parse("a,a,a,a,a") == ["a", "a", "a", "a", "a"]


def test_sep_by_empty():
    p = combinators.sep_by(parser.single("a"), parser.single(","))
    assert p.parse("") == []


def test_sep_by_repr():
    assert repr(combinators.sep_by(parser.single("a"),
                                   parser.single(","))) == \
        "sep_by(single('a'), single(','))"


def test_sep_by1():
    p = combinators.sep_by1(parser.single("a"), parser.single(","))
    assert p.parse("a,a,a,a,a") == ["a", "a", "a", "a", "a"]


def test_sep_by1_empty():
    p = combinators.sep_by1(parser.single("a"), parser.single(","))
    try:
        p.parse("")
    except parser.Unexpected as e:
        if e.expected != frozenset(["a"]) or e.actual != parser.EOF:
            raise
    else:
        assert False, "should have thrown exception"


def test_sep_by1_repr():
    assert repr(combinators.sep_by1(parser.single("a"),
                                    parser.single(","))) == \
        "sep_by1(single('a'), single(','))"


def test_between_repr():
    assert repr(combinators.between(parser.single("("),
                                    parser.single(")"),
                                    parser.single("a"))) == \
        "between(single('('), single(')'), single('a'))"


def test_try_basic():
    p = combinators.try_(parser.single("a"))
    assert p.parse("a") == "a"


def test_try_no_match():
    p = combinators.try_(parser.single("a"))
    try:
        p.parse("b")
    except parser.Unexpected as e:
        if e.expected != frozenset(["a"]) or e.actual != "b":
            raise
    else:
        assert False, "should have thrown exception"

    assert p.parse("a") == "a"


def test_try_choice_sequence():
    p = combinators.choice(
        combinators.try_(
            combinators.sequence(parser.single("a"), parser.single("b"))
        ),
        combinators.sequence(parser.single("a"), parser.single("a"))
    )
    assert p.parse("aa") == ["a", "a"]


def test_no_try():
    p = combinators.choice(
        combinators.sequence(parser.single("a"), parser.single("b")),
        combinators.sequence(parser.single("a"), parser.single("a"))
    )
    try:
        p.parse("aa")
    except parser.Unexpected as e:
        if e.expected != frozenset(["b", "a"]):
            raise
    else:
        assert False, "should have thrown exception"


def test_try_repr():
    assert repr(combinators.try_(parser.single("a"))) == "try_(single('a'))"


def test_mapf():
    assert combinators.mapf(parser.any_)(lambda x: "foo").parse(["bar"]) == \
        "foo"


def test_mapf_star():
    assert combinators.mapf_star(
        parser.unit(lambda: ["foo", "bar"]))(lambda x, y: x).parse([]) == "foo"
