from kessel import parser


def test_peek_state():
    state = parser.State(["a"])
    assert state.peek() == (0, "a")
    assert state.peek() == (0, "a")


def test_expect_state():
    state = parser.State([])
    try:
        state.expect(0, "a", "b")
    except parser.Unexpected as e:
        if e.index != 0 or e.expected != "a" or e.actual != "b":
            raise
    else:
        assert False, "should have thrown exception"


def test_error_state():
    state = parser.State([])
    try:
        state.error(0, "derp")
    except parser.ParseError as e:
        if e.index != 0 or e.message != "derp":
            raise
    else:
        assert False, "should have thrown exception"


def test_error_str():
    assert str(parser.ParseError(0, "hi")) == "hi at index 0"


def test_error_repr():
    assert repr(parser.error("hi")) == "error('hi')"


def test_expected_str():
    assert str(parser.Unexpected(0, frozenset(["hi"]), "bye")) == \
        "expected one of 'hi', got 'bye' at index 0"


def test_negate_repr():
    assert repr(parser.Unexpected.Negate(frozenset(["a", "b"]))) == \
        "none of 'a', 'b'"


def test_advance_state():
    state = parser.State(["a", "b"])
    assert state.advance() == (0, "a")
    assert state.advance() == (1, "b")


def test_single():
    hello = parser.single("hello")
    assert hello.parse(["hello"]) == "hello"


def test_single_repr():
    assert repr(parser.single("hello")) == "single('hello')"


def test_any():
    assert parser.any_.parse(["hello"]) == "hello"


def test_any_repr():
    assert repr(parser.any_) == "any_"


def test_nothing_expected():
    assert parser.Parser().parse(["hello"]) is None


def test_bad_single():
    hello = parser.single("hello")
    try:
        hello.parse(["goodbye"])
    except parser.Unexpected as e:
        if e.expected != frozenset(["hello"]) or e.actual != "goodbye":
            raise
    else:
        assert False, "should have thrown exception"


def test_one_of():
    hello = parser.one_of("hello")
    assert hello.parse(["hello"]) == "hello"


def test_bad_one_of():
    hello = parser.one_of("hello")
    try:
        hello.parse(["goodbye"])
    except parser.Unexpected as e:
        if e.expected != frozenset(["hello"]) or e.actual != "goodbye":
            raise
    else:
        assert False, "should have thrown exception"


def test_one_of_repr():
    assert repr(parser.one_of("hello")) == "one_of('hello')"


def test_none_of():
    hello = parser.none_of("hello")
    assert hello.parse(["goodbye"]) == "goodbye"


def test_bad_none_of():
    hello = parser.none_of("hello")
    try:
        hello.parse(["hello"])
    except parser.Unexpected as e:
        if e.expected != frozenset([parser.Unexpected.Negate(frozenset(["hello"]))]) or e.actual != "hello":
            raise
    else:
        assert False, "should have thrown exception"


def test_none_of_repr():
    assert repr(parser.none_of("hello")) == "none_of('hello')"


@parser.gen_parser
def concat(first, second):
    first2 = yield parser.single(first)
    second2 = yield parser.single(second)

    return first2, second2


def test_generator_concat_parser():
    p = concat("hello", "world")
    assert p.parse(["hello", "world"])


def test_gen_parser_repr():
    assert repr(concat) == "<gen_parser concat>"


@parser.gen_parser
def exception_parser():
    try:
        raise Exception
    except Exception:
        return "hello"
    yield


def test_generator_concat_parser_exception():
    assert exception_parser.parse([]) == "hello"


def test_eof_repr():
    assert repr(parser.EOF) == "end of file"


def test_eof():
    parser.eof.parse([])


def test_bad_eof():
    try:
        parser.eof.parse(["a"])
    except parser.Unexpected as e:
        if e.expected != frozenset([parser.EOF]) or e.actual != "a":
            raise
    else:
        assert False, "should have thrown exception"


def test_error():
    try:
        parser.error("a").parse([])
    except parser.ParseError as e:
        if e.message != "a":
            raise
    else:
        assert False, "should have thrown exception"


def test_unit():
    assert parser.unit(lambda: "a").parse([]) == "a"


def test_unit_repr():
    assert repr(parser.unit(lambda: "a")) == "unit(...)"


def test_no_forward():
    p = parser.forward()
    try:
        p.parse([])
    except ValueError:
        pass
    else:
        assert False, "should have thrown exception"


def test_forward():
    p = parser.forward()
    p.set(parser.single("hello"))
    assert p.parse(["hello"]) == "hello"


def test_forward_repr():
    assert repr(parser.forward()) == "forward(None)"
