"""
Base parsers.
"""

import functools

__all__ = ["State", "Unexpected", "Parser", "any_", "single", "none_of",
           "one_of", "gen_parser", "eof", "error", "unit", "ParseError", "EOF",
           "forward"]


class State:
    """
    A parser state enables peeking and advancing the stream by one item at
    once.
    """

    def __init__(self, xs):
        self._peek = None
        self._has_peek = False
        self.index = -1
        self._it = iter(xs)

    def advance(self):
        """
        Advance the stream position by a single item.
        """

        if self._has_peek:
            v = self._peek
            self._peek = None
            self._has_peek = False
        else:
            try:
                v = next(self._it)
            except StopIteration:
                v = EOF

        if v is not EOF:
            self.index += 1

        return self.index, v

    def peek(self):
        """
        Peek the stream by a single item.
        """

        if not self._has_peek:
            try:
                v = self._peek = next(self._it)
            except StopIteration:
                v = EOF
            else:
                self._has_peek = True
        else:
            v = self._peek

        return self.index + 1, v

    def error(self, index, message):
        """
        Raise a parser error.
        """

        raise ParseError(index, message)

    def expect(self, index, expected, actual):
        """
        Raise an expected error.
        """

        raise Unexpected(index, expected, actual)


class ParseError(Exception):
    """
    An exception thrown when parsing fails.
    """

    def __init__(self, index, message):
        self.index = index
        self.message = message

    def __str__(self):
        return "{} at index {}".format(self.message, self.index)


class Unexpected(ParseError):
    """
    This exception is raised when a parser encounters input that it does not
    expect.
    """

    class Negate:
        def __init__(self, values):
            self.values = values

        def __repr__(self):
            return "none of {}".format(
                ", ".join(repr(x) for x in sorted(self.values)))

        def __eq__(self, rhs):
            return self.values == rhs.values

        def __hash__(self):
            return hash(self.values)

    def __init__(self, index, expected, actual):
        self.index = index
        self.expected = expected
        self.actual = actual

    @property
    def message(self):
        return "expected one of {expected}, got {actual!r}".format(
            expected=", ".join(repr(x) for x in sorted(self.expected)),
            actual=self.actual,
        )


class Parser:
    """
    The base parser class defines some parser set-up routines and delegates
    parsing to a method named ``do_parse``, which will accept an iterator.
    """

    def parse(self, xs):
        return self.do_parse(State(xs))

    def do_parse(self, xs):
        """
        This method should be overridden to define custom parsing behavior.
        """

    def __add__(self, rhs):
        """
        Sequence two parsers together.
        """
        from .combinators import sequence
        return sequence(self, rhs)

    def __or__(self, rhs):
        """
        Alternate two parsers.
        """
        from .combinators import choice
        return choice(self, rhs)

    def __lshift__(self, rhs):
        """
        Sequence two parsers, discarding the result of the second.
        """
        from .combinators import sequence_l
        return sequence_l(self, rhs)

    def __rshift__(self, rhs):
        """
        Sequence two parsers, discarding the result of the first.
        """
        from .combinators import sequence_r
        return sequence_r(self, rhs)


class single(Parser):
    """
    A single parser accepts a single item, as-is.
    """

    def __init__(self, value):
        self.value = value

    def do_parse(self, state):
        i, x = state.peek()

        if x != self.value:
            state.expect(i, frozenset([self.value]), x)

        _, x = state.advance()
        return x

    def __repr__(self):
        return "single({value!r})".format(**self.__dict__)


class none_of(Parser):
    """
    A none-of parser accepts a single item that is not one of the given.
    """

    def __init__(self, *values):
        self.values = frozenset(values)

    def do_parse(self, state):
        i, x = state.peek()

        if x in self.values:
            state.expect(i, frozenset([Unexpected.Negate(self.values)]), x)

        _, x = state.advance()

        return x

    def __repr__(self):
        return "none_of({values})".format(
            values=", ".join(repr(x) for x in self.values))


class one_of(Parser):
    """
    A one-of parser accepts a single item that is one of the given.
    """

    def __init__(self, *values):
        self.values = frozenset(values)

    def do_parse(self, state):
        i, x = state.peek()

        if x not in self.values:
            state.expect(i, self.values, x)

        _, x = state.advance()
        return x

    def __repr__(self):
        return "one_of({values})".format(
            values=", ".join(repr(x) for x in self.values))


class _any_(Parser):
    """
    An any parser accepts a single item.
    """

    def do_parse(self, state):
        _, x = state.advance()
        return x

    def __repr__(self):
        return "any_"

any_ = _any_()


class error(Parser):
    """
    A parser that will always throw an error when run.
    """

    def __init__(self, message):
        self.message = message

    def do_parse(self, state):
        i, _ = state.peek()
        raise ParseError(i, self.message)

    def __repr__(self):
        return "error({message!r})".format(**self.__dict__)


class unit(Parser):
    """
    A parser that will always return value of the invocation of the bound
    function when run.
    """

    def __init__(self, f):
        self.f = f

    def do_parse(self, state):
        return self.f()

    def __repr__(self):
        return "unit(...)"


class _EndOfFile:
    def __repr__(self):
        return "end of file"

EOF = _EndOfFile()

eof = single(EOF)


class gen_parser(Parser):
    """
    Turns generators into parsers.
    """

    def __init__(self, f, *args, **kwargs):
        self.f = f
        self.args = args
        self.kwargs = kwargs

    def __call__(self, *args, **kwargs):
        return gen_parser(self.f, *args, **kwargs)

    def do_parse(self, state):
        gen = self.f(*self.args, **self.kwargs)

        try:
            try:
                parser = next(gen)
            except Exception as e:
                parser = gen.throw(e)
        except StopIteration as e:
            return e.value

        while True:
            try:
                try:
                    parser = gen.send(parser.do_parse(state))
                except Exception as e:
                    parser = gen.throw(e)
            except StopIteration as e:
                return e.value

    def __repr__(self):
        return "<gen_parser {f.__qualname__}>".format(**self.__dict__)


class forward(Parser):
    """
    Forward declare a parser to be filled in later.
    """

    def __init__(self):
        self.parser = None

    def set(self, parser):
        """
        Fill in the parser.
        """

        self.parser = parser

    def do_parse(self, state):
        if self.parser is None:
            raise ValueError("forward declaration has not been filled")

        return self.parser.do_parse(state)

    def __repr__(self):
        return "forward({parser!r})".format(**self.__dict__)
