from dm.cli.makeconfig import DomainConfigMaker
import os

class ConfigMaker(DomainConfigMaker):

    template = """[DEFAULT]
# Filesystem root (defaults to config file folder, or the config
# file folder's parent folder in the case where the config file
# folder's name is either 'etc' or 'config'). If other values 
# inside this configuration file substitute 'master_dir' with the
# syntax '%(master_dir)s' then the correct path must be set here.
#master_dir = /path/to/kforge

# Path to folder for KForge project service
# files (default: %(master_dir)s/var/project).
#project_data_dir = %(master_dir)s/var/project

# Name of KForge service (default: KForge).
#service_name = KForge

# Server name of KForge service (defaults to platform hostname).
#domain_name = kforge.your.domain

# Mode of operation (default: production).
#system_mode = production

# Salt for password digests (default: not-a-secret). Please note,
# this value must be carried forward when migrating data otherwise
# users will not be able to authenticate with their true passwords.
#password_digest_secret = not-a-secret

[db]
# When type is sqlite, set name to be a file path. Otherwise
# for postgres (and mysql) set a normal database name and
# set the user, password and host if necessary.

# Database type (default: sqlite).
#type = sqlite

# Database name (default for sqlite: %(master_dir)s/var/sqlite.db
# otherwise, default for other database types: kforge).
#name = %(master_dir)s/var/sqlite.db

# Database user (default: kforge).
#user = kforge

# Database password (default: ).
#pass = 

# Database host (default: localhost).
#host = localhost

[django]
# Path to folder of template files (default: %(master_dir)s/templates).
#templates_dir = %(master_dir)s/templates

# Secret to seed session key generation (default: not-a-secret).
#secret_key = not-a-secret 

[email]
# Enable email (default: ).
#enable_sending = on

# Send notifications of changes to administrators (default: ).
#notify_changes = on

# Email address for sending service emails (defaults to good guess).
#service_address = kforge-noreply@your.domain.com

# Socket for connections with SMTP server (default: localhost).
#smtp_host = localhost

[environment]
# Timezone, see platform's zoneinfo database (defaults to environment).
#timezone = Europe/London

[feed]
# Maximum number of items in feed (default: 100).
#length = 100

# Maximum number of items in feed summary (default: 25).
#length_summary = 25

[logging]
# Path to KForge log file (default: %(master_dir)s/var/log/kforge.log).
#log_file = %(master_dir)s/var/log/kforge.log

# Log level ERROR, WARNING, or INFO (default: WARNING).
#level = WARNING

[memos]
# Enable memoization of access controller results (default: ).
#enabled = on

# Maximum number of memos to store (default: 3000).
#limit = 3000

# Maximum time (seconds) validity of memos (default: 30).
#expire = 30

[virtualenv]
# Path to bin folder of virtual environment (required within virtualenv).
#bin_dir = %(master_dir)s/bin

[www]
# Suspend reloading Apache (default: ).
no_reload_apache = on

# Shell command to reload Apache (default: sudo /etc/init.d/apache2 graceful).
# Note, adding "www-data ALL=(ALL) NOPASSWD: /etc/init.d/apache2 graceful" to
# you sudoers file may help to make reloading work smoothly. :-) Of course,
# change "www-data" to whichever user the process will run as.
#reload_apache = sudo /etc/init.d/apache2 graceful

# Path to folder with css, images, and scripts (default: %(master_dir)s/media).
#media_dir = %(master_dir)s/media

# Path to auto-generated Apache configuration file (required).
#apache_config_file = %(master_dir)s/var/httpd-autogenerated.conf

# Path to the auto-generated WSGI file (default: %(master_dir)s/wsgi/kforge.wsgi).
#wsgi_file = %(master_dir)s/wsgi/kforge.wsgi

# Apache python module (default: mod_wsgi). Check the chosen module is
# enabled (a2enmod) and restart Apache. The legacy alternative is mod_python.
#python_module = mod_wsgi

# Seconds delay (default: 12) before redirecting after a service has been
# created, updated or deleted, and when mod_wsgi is in use, and when KForge 
# is configured to reload Apache. A positive integer value will cause the
# subsequent page to be loaded after this delay, by returning an HTTP 200 OK
# response with a Refresh header. A delay of 0 will cause such redirects to
# be HTTP 302 Found, which causes the subsequent page to be loaded
# immediately. It has been found that without a sufficient delay, users may
# be presented with (and discouraged by) a 'Service Temporarily Unavailable'
# notice from Apache. The value has no effect if mod_python is in use, or if
# reloading of Apache configuration by KForge is suspended.
#wsgi_redirect_delay = 12


## ********************************************************************
## Plugins
## ********************************************************************

[git]
# Git executable (default: git).
#admin_script = git

# Path to HTTP backend script (default: /usr/lib/git-core/git-http-backend).
#http_backend_script = /usr/lib/git-core/git-http-backend

# Path to GitWeb script (default: /usr/lib/cgi-bin/gitweb.cgi).
#gitweb_script = /usr/lib/cgi-bin/gitweb.cgi

# Path to GitWeb static media (default: /usr/share/gitweb/static).
#gitweb_static = /usr/share/gitweb/static


[joomla]
# Path to master Joomla installation to be used as a template (required).
master_path = /usr/share/joomla_1.5

# Shell command to dump the Joomla database to stdout (required).
backup_command = mysqldump --user=JOOMLADBUSER --password=JOOMLADBPASS JOOMLADBNAME

[mailman]
# Shell command (without arguments) to create a list (default: sudo newlist).
#newlist = sudo newlist

# Shell command (without arguments) to delete a list (default: sudo rmlist).
#rmlist = sudo rmlist

# Password that will be used for new mailing lists (required).
list_admin_password = change_this_password_immediately

# Domain name for Mailman site.
# set this to the fully qualified domain name for your mailing lists
# example: lists.domain.com (NB: no http and no trailing url)
# NB: expect that mailman web interface will have been mounted there
# If this is a problem (e.g. you want to have the email host and web host to be
# different) you will have to make some modifications to the mailman plugin
# code.
mailman_url = lists.yourdomain.com

#[mercurial]
#admin_script = hg

[moin]
# Path to MoinMoin folder containing the htdocs folder (default: /usr/share/moin).
#system_path = /usr/share/moin

# Version name used to locate MoinMoin files (default: 193)
#version = 193

[ssh]
# Commands and keys are written into this file.
#authorized_keys_file = ~/.ssh/authorized_keys

# The SSH user name (used to indicate how to access services).
#user_name = %(user_name)s 

# The SSH host (used to indicate how to access services).
#host_name = %(domain_name)s

[svn]
# Use mod_python to run access control for Subversion via HTTP (default: ).
# When using mod_wsgi to run the KForge software (see python_module in the
# [www] section), access control for Subversion via HTTP causes a password
# prompt to be shown during permitted anonymous checkout operations. However,
# When using mod_wsgi to run the KForge software in Apache, using mod_python
# to run the access control for Subversion via HTTP allows KForge to suppress
# the password prompt during permitted anonymous checkout operations.
# If this value is set, the Apache module mod_python will need to be enabled.
#dav_mod_python_access_control = on

# Path to the ViewVC Python library (default: /usr/lib/viewvc/lib).
#viewvc_lib = /usr/lib/viewvc/lib

# Path to the ViewVC templates folder (default: /etc/viewvc/templates).
#viewvc_template_dir = /etc/viewvc/templates

# Path to the ViewVC media folder (default: /usr/share/viewvc/docroot).
#viewvc_media_dir = /usr/share/viewvc/docroot

[trac]
# Trac admin script (default: trac-admin).
#admin_script = trac-admin

# Path to Trac templates folder (required trac < v0.11).
#templates_path = /usr/share/trac/templates

# Path to Trac htdocs folder (optional).
#htdocs_path = /usr/share/trac/htdocs

[wordpress]
# Path to Wordpress folder containing template install (default: /usr/share/wordpress).
#master_path = /usr/share/wordpress

# Shell command to dump the Wordpress db to stdout (required).
backup_command = mysqldump --user=WORDPRESSDBUSER --password=WORDPRESSDBPASS WORDPRESSDBNAME

"""

    def __init__(self, systemName='kforge'):
        super(ConfigMaker, self).__init__(systemName=systemName)

    def addOptions(self, parser):
        super(ConfigMaker, self).addOptions(parser)

        parser.add_option(
            '--project-data-dir',
            dest='projectDataDir',
            help="""The folder to be used for project service data.
Please note, it is recommended to set a path to a directory outside of
the master folder (--master-dir) since new versions will be installed
in isolation from previous versions, but they will need to use the same
project service data folder, which may cause trouble if moved once
created. That is, if you put your project service data inside your first
installation, subsequent installations will be forced to refer to a path
in an older installation.""")

        parser.add_option(
            '--trac-admin',
            dest='tracAdmin',
            help="""The path to your trac-admin script.""")

        parser.add_option(
            '--viewvc-dir',
            dest='viewvcDir',
            help="""The path to your ViewVC installation.""")

    def addOptionLines(self, options, optionLines):
        super(ConfigMaker, self).addOptionLines(options, optionLines)
        if options.projectDataDir:
            optionLines.append('[DEFAULT]')
            optionLines.append('project_data_dir = %s' % os.path.abspath(options.projectDataDir))
        if options.viewvcDir:
            optionLines.append('[svn]')
            optionLines.append('viewvc_lib = %s' % os.path.abspath(os.path.join(options.viewvcDir, 'lib')))
            optionLines.append('viewvc_template_dir = %s' % os.path.abspath(os.path.join(options.viewvcDir, 'templates')))
            optionLines.append('viewvc_media_dir = %s' % os.path.abspath(os.path.join(options.viewvcDir, 'templates', 'docroot')))
        if options.tracAdmin:
            optionLines.append('[trac]')
            optionLines.append('admin_script = %s' % options.tracAdmin)

