var toolbar = [
  {name: 'bold', action: toggleBold},
  {name: 'italic', action: toggleItalic},
  '|',

  {name: 'quote', action: toggleBlockquote},
  {name: 'unordered-list', action: toggleUnOrderedList},
  {name: 'ordered-list', action: toggleOrderedList},
  '|',

  {name: 'link', action: drawLink},
  {name: 'image', action: drawImage},
  '|',

  {name: 'info', action: 'http://lab.lepture.com/editor/markdown'},
  {name: 'preview', action: togglePreview},
  {name: 'fullscreen', action: toggleFullScreen}
];

/**
 * Interface of Editor.
 */
function Editor(options) {
  options = options || {};

  if (options.element) {
    this.element = options.element;
  }

  options.toolbar = options.toolbar || Editor.toolbar;
  // you can customize toolbar with object
  // [{name: 'bold', shortcut: 'Ctrl-B', className: 'icon-bold'}]

  if (!options.hasOwnProperty('status')) {
    options.status = ['lines', 'words', 'cursor'];
  }

  this.options = options;

  // If user has passed an element, it should auto rendered
  if (this.element) {
    this.render();
  }
}

/**
 * Default toolbar elements.
 */
Editor.toolbar = toolbar;

/**
 * Default markdown render.
 */
Editor.markdown = function(text) {
  if (window.marked) {
    // use marked as markdown parser
    return marked(text);
  }
};

/**
 * Render editor to the given element.
 */
Editor.prototype.render = function(el) {
  if (!el) {
    el = this.element || document.getElementsByTagName('textarea')[0];
  }

  if (this._rendered && this._rendered === el) {
    // Already rendered.
    return;
  }

  this.element = el;
  var options = this.options;

  var self = this;
  var keyMaps = {};

  for (var key in shortcuts) {
    (function(key) {
      keyMaps[fixShortcut(key)] = function(cm) {
        shortcuts[key](self);
      };
    })(key);
  }

  keyMaps["Enter"] = "newlineAndIndentContinueMarkdownList";
  keyMaps['Tab'] = 'tabAndIndentContinueMarkdownList';
  keyMaps['Shift-Tab'] = 'shiftTabAndIndentContinueMarkdownList';

  this.codemirror = CodeMirror.fromTextArea(el, {
    mode: 'markdown',
    theme: 'paper',
    tabSize: '2',
    indentWithTabs: true,
    lineNumbers: false,
    autofocus: true,
    extraKeys: keyMaps
  });

  if (options.toolbar !== false) {
    this.createToolbar();
  }
  if (options.status !== false) {
    this.createStatusbar();
  }

  this._rendered = this.element;
};

Editor.prototype.createToolbar = function(items) {
  items = items || this.options.toolbar;

  if (!items || items.length === 0) {
    return;
  }

  var bar = document.createElement('div');
  bar.className = 'editor-toolbar';

  var self = this;

  var el;
  self.toolbar = {};

  for (var i = 0; i < items.length; i++) {
    (function(item) {
      var el;
      if (item.name) {
        el = createIcon(item.name, item);
      } else if (item === '|') {
        el = createSep();
      } else {
        el = createIcon(item);
      }

      // bind events, special for info
      if (item.action) {
        if (typeof item.action === 'function') {
          el.onclick = function(e) {
            item.action(self);
          };
        } else if (typeof item.action === 'string') {
          el.href = item.action;
          el.target = '_blank';
        }
      }
      self.toolbar[item.name || item] = el;
      bar.appendChild(el);
    })(items[i]);
  }

  var cm = this.codemirror;
  cm.on('cursorActivity', function() {
    var stat = getState(cm);

    for (var key in self.toolbar) {
      (function(key) {
        var el = self.toolbar[key];
        if (stat[key]) {
          el.className += ' active';
        } else {
          el.className = el.className.replace(/\s*active\s*/g, '');
        }
      })(key);
    }
  });

  var cmWrapper = cm.getWrapperElement();
  cmWrapper.parentNode.insertBefore(bar, cmWrapper);
  return bar;
};

Editor.prototype.createStatusbar = function(status) {
  status = status || this.options.status;

  if (!status || status.length === 0) return;

  var bar = document.createElement('div');
  bar.className = 'editor-statusbar';

  var pos, cm = this.codemirror;
  for (var i = 0; i < status.length; i++) {
    (function(name) {
      var el = document.createElement('span');
      el.className = name;
      if (name === 'words') {
        el.innerHTML = '0';
        cm.on('update', function() {
          el.innerHTML = wordCount(cm.getValue());
        });
      } else if (name === 'lines') {
        el.innerHTML = '0';
        cm.on('update', function() {
          el.innerHTML = cm.lineCount();
        });
      } else if (name === 'cursor') {
        el.innerHTML = '0:0';
        cm.on('cursorActivity', function() {
          pos = cm.getCursor();
          el.innerHTML = pos.line + ':' + pos.ch;
        });
      }
      bar.appendChild(el);
    })(status[i]);
  }
  var cmWrapper = this.codemirror.getWrapperElement();
  cmWrapper.parentNode.insertBefore(bar, cmWrapper.nextSibling);
  return bar;
};

/**
 * Get or set the text content.
 */
Editor.prototype.value = function(val) {
  if (val) {
    this.codemirror.getDoc().setValue(val);
    return this;
  } else {
    return this.codemirror.getValue();
  }
};


/**
 * Bind static methods for exports.
 */
Editor.toggleBold = toggleBold;
Editor.toggleItalic = toggleItalic;
Editor.toggleBlockquote = toggleBlockquote;
Editor.toggleUnOrderedList = toggleUnOrderedList;
Editor.toggleOrderedList = toggleOrderedList;
Editor.drawLink = drawLink;
Editor.drawImage = drawImage;
Editor.undo = undo;
Editor.redo = redo;
Editor.togglePreview = togglePreview;
Editor.toggleFullScreen = toggleFullScreen;

/**
 * Bind instance methods for exports.
 */
Editor.prototype.toggleBold = function() {
  toggleBold(this);
};
Editor.prototype.toggleItalic = function() {
  toggleItalic(this);
};
Editor.prototype.toggleBlockquote = function() {
  toggleBlockquote(this);
};
Editor.prototype.toggleUnOrderedList = function() {
  toggleUnOrderedList(this);
};
Editor.prototype.toggleOrderedList = function() {
  toggleOrderedList(this);
};
Editor.prototype.drawLink = function() {
  drawLink(this);
};
Editor.prototype.drawImage = function() {
  drawImage(this);
};
Editor.prototype.undo = function() {
  undo(this);
};
Editor.prototype.redo = function() {
  redo(this);
};
Editor.prototype.togglePreview = function() {
  togglePreview(this);
};
Editor.prototype.toggleFullScreen = function() {
  toggleFullScreen(this);
};
