# coding=utf-8
# Filename: app.py
"""
PipeInspector

Usage:
    pipeinspector -i <file>
    pipeinspector (-h | --help)
    pipeinspector --version

Options:
    -h --help       Show this screen.
    -i <file>       Input file.

"""
from __future__ import division, absolute_import, print_function

import os

import urwid

from pipeinspector.gui import MainFrame
from pipeinspector.settings import UI
from km3pipe.pumps import EvtPump, DAQPump, AanetPump

__version__ = "1.0.0"


def handle_input(input):
    """Handle any unhandled input."""
    if input in UI.keys['escape']:
        raise urwid.ExitMainLoop

def filter_input(keys, raw):
    """Adds fancy mouse wheel functionality and VI navigation to ListBox"""
    if len(keys) == 1:
        if keys[0] in UI.keys['up']:
            keys[0] = 'up'
        elif keys[0] in UI.keys['down']:
            keys[0] = 'down'
        elif len(keys[0]) == 4 and keys[0][0] == 'mouse press':
            if keys[0][1] == 4:
                keys[0] = 'up'
            elif keys[0][1] == 5:
                keys[0] = 'down'
    return keys


def get_pump(input_file):
    extension = os.path.splitext(input_file)[1][1:]
    if extension == 'evt':
        pump = EvtPump(filename=input_file)
    elif extension == 'dat':
        pump = DAQPump(filename=input_file)
    elif extension == 'root':
        pump = AanetPump(filename=input_file)
    else:
        raise SystemExit("No pump found for '{0}' files.".format(extension))
    return pump


def main():
    from docopt import docopt
    arguments = docopt(__doc__, version=__version__)
    input_file = arguments['-i']
    pump = get_pump(input_file)
    main_frame = MainFrame(pump)
    #main_frame.header.set_text("Inspecting {0}".format(input_file))
    loop = urwid.MainLoop(main_frame, UI.palette,
                      input_filter=filter_input,
                      unhandled_input=handle_input)
    loop.run()


if __name__ == '__main__':
    main()
