import os
import platform
import re
import shutil
import sys

from setuptools.command.build_ext import build_ext
from setuptools.command.install_lib import install_lib
from setuptools import Extension
from setuptools import setup
from subprocess import call

generated_lib = None


class wurf_install_lib(install_lib):
    def install(self):
        target = os.path.abspath(self.install_dir)
        if not os.path.isdir(target):
            os.makedirs(target)

        global generated_lib
        output_file = os.path.join(target, os.path.basename(generated_lib))
        if os.path.exists(output_file):
            os.remove(output_file)
        shutil.move(generated_lib, target)
        print('WRITING LIBRARY: {}'.format(output_file))
        return [output_file]


class wurf_build_ext(build_ext):
    def run(self):

        # Clone project
        call([
            'git',
            'clone',
            'git@github.com:steinwurf/kodo-python.git',
            'kodo-python'])

        cwd = os.getcwd()
        os.chdir(os.path.join(cwd, 'kodo-python'))

        # Checkout the right version
        call(['git', 'checkout', '6.0.0'])

        # We should always build the python bindings with the architecture
        # of the used python executable.
        python_architecture = platform.architecture()[0]

        # Mapping from the python architecture to a suitable mkspec.
        cxx_mkspec = {
            '32bit': 'cxx_default_x86',
            '64bit': 'cxx_default_x64',
        }[python_architecture]

        install_path = 'installed_binaries'
        build_cmd = [
            sys.executable,
            'waf',
            'configure',
            '--options=install_path={},cxx_mkspec={}'.format(
                install_path,
                cxx_mkspec),
            'build', 'install']

        call(build_cmd)
        # Find the created library file
        # This can be either:
        #   [python_lib].so
        #   [python_lib].pyd
        #   [python_lib].cpython-34m.so
        # and possibly more.
        regex = '{}\.(so|pyd|.*\.so)$'.format('kodo')

        global generated_lib
        for filename in os.listdir(install_path):
            if re.match(regex, filename):
                generated_lib = os.path.join(
                    'kodo-python', install_path, filename)
        os.chdir(cwd)

setup(
    name='kodo',
    version='6.0.0',
    ext_modules=[Extension('kodo', sources=[])],
    description='RLNC Network Coding Library',
    long_description=open('README.txt').read(),
    license='Steinwurf Research License',
    author='Steinwurf ApS',
    author_email='pip@steinwurf.com',
    url='False',
    keywords='steinwurf network coding rlnc random linear network coding ',
    platforms=[
        'Windows',
        'Linux',
        'MacOS X',
        ],
    classifiers=[
        'Development Status :: 4 - Beta',
        'Intended Audience :: Developers',
        'Intended Audience :: Science/Research',
        'Intended Audience :: Telecommunications Industry',
        'License :: Free For Educational Use',
        'Operating System :: MacOS :: MacOS X',
        'Operating System :: Microsoft :: Windows',
        'Operating System :: POSIX :: Linux',
        'Programming Language :: C++',
        'Programming Language :: Python',
        'Programming Language :: Python :: 2',
        'Programming Language :: Python :: 2.7',
        'Programming Language :: Python :: 3',
        'Programming Language :: Python :: 3.4',
        ],
    cmdclass={'build_ext': wurf_build_ext, 'install_lib': wurf_install_lib}
)
