# -*- coding: utf-8 -*-
"""
This module provides the tornado RequestHandler base class.

Usage::

        from krux.tornado.handlers import RequestHandler


        class MyHandler(RequestHandler):
            def get(self):
                self.write("Hello, world!")

If you need to over-ride :py:meth:`~RequestHandler.prepare` or
:py:meth:`~RequestHandler.on_finish`, you *must* call :py:func:`python:super`
or stats will not work correctly::

        class MyHandler(RequestHandler):
            def prepare(self):
                # Your code goes here.
                super(MyHandler, self).prepare()
                # Your code goes here.

            def on_finish(self):
                # Your code goes here.
                super(MyHandler, self).on_finish()
                # Your code goes here.
"""

######################
# Standard Libraries #
######################
from __future__ import absolute_import

import time

#########################
# Third Party Libraries #
#########################
import tornado.web

try:
    import simplejson as json
except ImportError:
    import json

######################
# Internal Libraries #
######################


class RequestHandler(tornado.web.RequestHandler):
    """
    Base request handler for Krux applications, automatically counts and times
    requests, and sends those stats to the stats service.
    """
    def prepare(self):
        """
        Start a timer for the current request.
        """
        # Because we are stopping the timer in a separate method (on_finish),
        # we have to track the time manually instead of using the stats client
        # methods.
        self.start_time = time.time()

    def get_stats_key(self):
        """
        Return the stats key (statsd-style) for the current request.

        This key is generated by taking the path of the request and stripping
        out any path arguments (regular expression groupings in the handler
        specification) and path keyword arguments (named regular expression
        groupings in the handler specification). From this, '/' characters are
        replaced with '.' characters to give us a statsd-style path.
        """
        # We want to strip both unnamed groupings (path_args) and named
        # groupings (path_kwargs.values()).
        path_args = self.path_args + self.path_kwargs.values()

        path = self.request.path
        for arg in path_args:
            path = path.replace(arg, '')
        path = path.replace('//', '/').strip('/').replace('/', '.')

        if not (200 <= self.get_status() < 400):
            # To avoid littering graphite with tons of keys, we record errors
            # at the top level.
            return path.partition('.')[0]
        else:
            return path

    def send_stats(self):
        """
        Send stats for the current request. Must be called after the request is
        finished (i.e. by on_finish or send_error).
        """
        self.application.logger.debug('Sending stats.')

        stats_key = self.get_stats_key()
        method = self.request.method.upper()

        # Increment the request count for this request type and status.
        self.application.endpoint_stats.incr(
            '.'.join((stats_key, method, str(self.get_status())))
        )

        # Send timing information for this request, and log the time for
        # debugging.
        #
        # We multiply by 1000 so our unit is milliseconds, and then round it
        # to the nearest integer for consistency with historical code.
        delta = int(round((time.time() - self.start_time) * 1000))
        self.application.endpoint_stats.timing(
            '.'.join((stats_key, method)),
            delta
        )
        self.application.logger.debug(
            '*** Stats Timer: %s: %sms', stats_key, delta
        )

    def on_finish(self):
        """
        Send stats for the current request by calling :py:func:`send_stats`.
        """
        self.send_stats()


class ErrorHandler(RequestHandler):
    """
    Generates an error response with status_code for all requests while also
    tracking stats.
    """
    def initialize(self):
        """
        This handler should only be used as a fallback for non-existing URLs.
        """
        self.set_status(404)

    def prepare(self):
        """
        Raise an HTTP error.
        """
        super(ErrorHandler, self).prepare()
        raise tornado.web.HTTPError(self.get_status())

    def get_stats_key(self):
        """
        To avoid littering our stats with a bunch of random paths, we use
        _catchall for the stats key for errors.
        """
        return '.'.join((self.application.endpoint, '_catchall'))


class StatusHandler(RequestHandler):
    """
    Basic status handler, dispatches to the application's status handler, if
    one is defined.
    """
    def status(self):
        """
        Call the applications's ``status`` method, if one exists, and send the
        result to the client. If no ``status`` method exists, send an empty
        result.
        """
        app_status_handler = getattr(self.application, 'status', None)
        if app_status_handler is not None:
            self.send_status(app_status_handler())
        else:
            self.send_status({
                'state': 'ok',
                'status': 'Application is responding with default '
                'status handler.',
                'version': getattr(self.application, 'version', None)
            })

    def send_status(self, status):
        """
        Send ``status`` (JSON-encoded) to the client.
        """
        try:
            json_status = json.dumps(status)
        except ValueError as err:
            json_status = 'Error JSON encoding status: %s' % err
            self.application.logger.debug(json_status)

        self.write(json_status)

    def set_default_headers(self):
        """
        Set the default headers for each request.
        """
        self.set_header('Content-Type', 'application/json')

    def get(self):
        """
        Handle GET requests by returning the application status as the
        response.
        """
        self.set_status(200)
        self.status()

    def head(self):
        """
        Handle HEAD requests by sending application headers.
        """
        self.set_status(200)


class NoContentHandler(RequestHandler):
    """
    Returns 204 No Content for all requests.
    """
    def get(self, *args, **kwargs):
        """
        Respond to GET requests with 204 No Content
        """
        self.set_status(204)

    def head(self, *args, **kwargs):
        """
        Respond to HEAD requests with 204 No Content
        """
        self.set_status(204)
