#coding=utf8

# Copyright © 2012 Tim Radvan
# 
# This file is part of Kurt.
# 
# Kurt is free software: you can redistribute it and/or modify it under the 
# terms of the GNU Lesser General Public License as published by the Free 
# Software Foundation, either version 3 of the License, or (at your option) any 
# later version.
# 
# Kurt is distributed in the hope that it will be useful, but WITHOUT ANY 
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR 
# A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more 
# details.
# 
# You should have received a copy of the GNU Lesser General Public License along 
# with Kurt. If not, see <http://www.gnu.org/licenses/>.

"""Compiles a folder structure generated by decompile.py to a Scratch project.
Imports PNG or JPG images.
Scripts must be scratchblocks format txt files.

    Usage: python compile.py "path/to/project files/"
"""

import time
import os, sys
from os.path import join as join_path
from os.path import split as split_path

import codecs
def open(file, mode="r"):
    log(file)
    return codecs.open(file, mode, "utf-8")


try:
    import kurt
except ImportError: # try and find kurt directory
    path_to_file = join_path(os.getcwd(), __file__)
    path_to_lib = split_path(split_path(path_to_file)[0])[0]
    sys.path.append(path_to_lib)

from kurt import *



class InvalidFile(Exception):
    def __init__(self, path, error):
        self.path = path
        self.error = error
        Exception.__init__(self, error)

class FileNotFound(Exception):
    pass
    
class FileExists(Exception):
    pass



def log(msg, newline=True):
    if newline:
        print msg
    else:
        print msg, 


def split_filename_number(filename):
    parts = filename.split(' ')
    try:
        number = int(parts[0])
    except ValueError:
        raise InvalidFile(filename, "Name must start with number")
    
    name = " ".join(parts[1:])
    
    return (number, name)


def read_costume_file(path):
    try:
        f = open(path)
    except IOError:
        return []
    
    line_count = 0
    costumes = []
    while 1:
        line = f.readline()
        line_count += 1
        if not line:
            return costumes
        
        line = line.strip()
        if line:
            filename = line.strip()
            costume = {
                "filename": filename,
            }
            
            while 1:
                line = f.readline()
                
                if not line.startswith("#"):                    
                    line_count += 1
                    if not line:
                        return costumes
                    
                    line = line.strip()
                    if not line:
                        break
                                        
                    parts = line.split(":")
                    if len(parts) == 2:
                        (key, value) = parts
                        costume[key] = value
                    elif len(parts) == 1:
                        key = parts[0]
                        costume[key] = None
                    else:
                        raise InvalidFile(path, "Line %i" % line_count)

            costumes.append(costume)
    



def read_script_file(morph, path):
    file = open(path)
    
    settings = {
        "pos": "(20, 20)",
    }
    
    while 1:
        try:
            line = file.readline()
        except EOFError:
            raise InvalidFile(path, "no blocks found")
        line = line.strip()
        
        if line:
            try:
                (name, value) = line.split(":")
            except ValueError:
                raise InvalidFile(path, "invalid line: "+line)
            
            name = name.strip().lower()
            value = value.strip()
            for setting in settings.keys():
                if name.startswith(setting):
                    settings[setting] = value
                    break
            else:
                settings[name] = value
        
        else:
            break
    
    pos = Point.from_string(settings["pos"])

    data = ""
    while 1:
        more_data = file.read()
        if not more_data: break
        data += more_data    
    
    data = str(data)
    
    try:
        blocks = parse_scratchblocks(data)
    except ParseError, e:
        raise InvalidFile(path, e)
    
    script = Script(morph, pos, blocks)

    file.close()
    
    return script



def import_sprite(project_dir, sprite_name):
    sprite_dir = join_path(project_dir, sprite_name)
    is_stage = (sprite_name == "00 Stage")
    number = 0
    if not is_stage:
        (number, sprite_name) = split_filename_number(sprite_name)    
    
    log("* "+sprite_name, False)
    start_time = time.time()
    
    if is_stage:
        sprite = ScratchStageMorph()
    else:
        sprite = ScratchSpriteMorph()
        sprite.name = sprite_name
    
    # Scripts
    scripts_dir = join_path(sprite_dir, "scripts")
    script_names = os.listdir(scripts_dir)
    
    scripts = []
    for script_name in script_names:
        script_path = join_path(scripts_dir, script_name)
        script = read_script_file(sprite, script_path)
        scripts.append(script)
    
    scripts.sort(key=lambda script: script.pos.y)
    sprite.scripts = scripts
    
    
    # Costumes/Backgrounds
    if is_stage:
        costumes_dir = join_path(sprite_dir, "backgrounds")
    else:
        costumes_dir = join_path(sprite_dir, "costumes")
    
    costumes = read_costume_file(costumes_dir+".txt")
    
    found_costumes = os.listdir(costumes_dir)
    found_costumes.sort()
    
    for costume in costumes:
        if costume["filename"] not in found_costumes:
            log("Couldn't find costume "+costume)
            costumes.remove(costume)
    
    for filename in found_costumes:
        for other in costumes:
            if filename == other["filename"]:
                break
        else:
            costumes.append({
                "filename": filename,
            })
    
    selected_costume = None
    for costume_args in costumes:
        if "selected" in costume_args:
            selected_costume = costume
            costume_args.pop("selected")

        filename = costume_args["filename"]
        #(index, costume_name) = split_filename_number(filename)

        costume_path = join_path(costumes_dir, filename)
        costume = ImageMedia.load(costume_path)
        costume.rotationCenter = Point.from_string(
            costume_args["rotationCenter"])

        print costume
        
        sprite.images.append(costume)
    
    if not selected_costume:
        selected_costume = costumes[-1]
    
    # TODO: Variables
    # TODO: Lists
    
    sprite_save_time = time.time() - start_time
    log(sprite_save_time)
    
    return (number, sprite)




def compile(project_dir, debug=True): # DEBUG: set to false
    if project_dir.endswith(".sb"):
        project_dir = project_dir[:-3]
    
    if project_dir.endswith(" files"):
        project_path = project_dir[:-6]
        
    else:
        project_path = project_dir
        project_dir += " files"
    
    if not os.path.exists(project_dir):
        raise FileNotFound(project_dir)
    
    log("Importing sprites...")
    
    sprite_names = os.listdir(project_dir)
    stage_name = "00 Stage"
    if stage_name not in sprite_names:
        raise FileNotFound(join_path(project_dir, stage_name))
    sprite_names.remove(stage_name)
    
    (_, stage) = import_sprite(project_dir, stage_name)
    
    sprites = [import_sprite(project_dir, name) for name in sprite_names]
    sprites.sort(key=lambda (n, s): n)
    sprites = [sprite for (number, sprite) in sprites]
    
    project = ScratchProjectFile.new(project_path+".sb")
    project.stage = stage
    project.sprites = sprites
    return project



if __name__ == '__main__':
    if len(sys.argv) <= 1:
        print __doc__
        exit()

    else:
        path = sys.argv[1]
    
        if path.endswith(".sb"):
            path = path[:-3]
        if path.endswith(" files"):
            path = path[:-6]
        
        if os.path.exists(path):
            raise FileExists(path)

        try:
            project = compile(path)
            project.save()

        except InvalidFile, e:
            print "Invalid file: %s" % e.path
            print e.error
            exit(2)

        except FileNotFound, e:
            print "File missing: %r" % str(e)
            exit(2)
