# Copyright (C) 2012 Linaro Limited
#
# Author: Zygmunt Krynicki <zygmunt.krynicki@linaro.org>
#
# This file is part of LAVA Fast Models
#
# LAVA Fast Models is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License version 3
# as published by the Free Software Foundation
#
# LAVA Fast Models is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with LAVA Fast Models.  If not, see <http://www.gnu.org/licenses/>.

from unittest2 import TestCase

from lava.nfs.backend import (
    Client,
    ClientList,
    Entry,
    EntryList,
    ExportsSerializer,
    Service,
)

class EntryTests(TestCase):

    def setUp(self):
        super(EntryTests, self).setUp()
        self.entry_list = EntryList([
            {
                'path': '/srv/foo/',
                'clients': [{
                    'client': '192.168.1.41',
                    'options': ['no_root_squash', 'async', 'rw']
                }]
            }
        ])
        self.entry = self.entry_list[0]

    def test_validness(self):
        self.assertEqual(self.entry.validate(), None)

    def test_export_cls(self):
        self.assertIsInstance(self.entry, Entry)

    def test_client_list_cls(self):
        self.assertIsInstance(self.entry.clients, ClientList)

    def test_client_cls(self):
        self.assertIsInstance(self.entry.clients[0], Client)

    def test_simple_nfs_export(self):
        observed = unicode(self.entry)
        expected = u"/srv/foo/ 192.168.1.41(no_root_squash,async,rw)"
        self.assertEqual(observed, expected)


class EntryListTests(TestCase):

    def setUp(self):
        super(EntryListTests, self).setUp()
        self.entry1 = {
            'path': '/srv/foo/',
            'clients': [{
                'client': '192.168.1.41',
                'options': ['no_root_squash', 'async', 'rw']
            }]
        }
        self.entry2 = {
            'path': '/srv/bar/',
            'clients': [{
                'client': '192.168.1.42',
                'options': ['no_root_squash', 'async', 'rw']
            }]
        }
        self.entry_list = EntryList([])

    def test_valid_when_empty(self):
        self.assertEqual(self.entry_list.validate(), None)

    def test_valid_when_not_empty(self):
        self.entry_list.append(self.entry1)
        self.assertEqual(self.entry_list.validate(), None)

    def test_append_works(self):
        expected = [self.entry1]
        self.entry_list.append(self.entry1)
        observed = self.entry_list.value
        self.assertEqual(observed, expected)

    def test_append_works_twice(self):
        expected = [self.entry1, self.entry2]
        self.entry_list.append(self.entry1)
        self.entry_list.append(self.entry2)
        observed = self.entry_list.value
        self.assertEqual(observed, expected)

    def test_append_bumps_revision(self):
        before = self.entry_list.revision
        self.entry_list.append(self.entry1)
        after = self.entry_list.revision
        self.assertNotEqual(before, after)


class ExportsSerializerTests(TestCase):

    def setUp(self):
        super(ExportsSerializerTests, self).setUp()
        self.serializer = ExportsSerializer

    def test_dumps_is_smart(self):
        self.assertRaises(TypeError, self.serializer.dumps, [])

    def test_dumps_works(self):
        obj = EntryList([
            {
                'path': '/zzz/',
                'clients': [
                    {
                        'client': '192.168.1.42',
                        'options': ['no_root_squash', 'async', 'rw']
                    }
                ]
            },
            {
                'path': '/aaa/',
                'clients': [
                    {
                        'client': '192.168.1.43',
                        'options': ['no_root_squash', 'async', 'rw']
                    }
                ]
            }
        ])
        expected = (
            "# DO NOT EDIT!\n"
            "# This file was automatically generated by the LAVA NFS service\n"
            "# To manipulate it, please use `lava nfs export` and `lava nfs unexport`\n"
            "\n"
            "/aaa/ 192.168.1.43(no_root_squash,async,rw)\n"
            "/zzz/ 192.168.1.42(no_root_squash,async,rw)\n"
        )
        observed = self.serializer.dumps(obj)
        self.assertEqual(observed.splitlines(), expected.splitlines())

    def test_loads_is_not_implemented(self):
        self.assertRaises(NotImplementedError, self.serializer.loads, "")
