# Copyright 2004-2009 Canonical Ltd.  All rights reserved.
#
# This file is part of lazr.batchnavigator
#
# lazr.batchnavigator is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# lazr.batchnavigator is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
# License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with lazr.batchnavigator. If not, see <http://www.gnu.org/licenses/>.

from zope.interface import Interface, Attribute

__metaclass__ = type


class InvalidBatchSizeError(AssertionError):
    """Received a batch parameter that exceed our configured max size."""

class IBatchNavigator(Interface):
    """A batch navigator for a specified set of results."""

    batch = Attribute("The IBatch for which navigation links are provided.")

    heading = Attribute(
        "The heading describing the kind of objects in the batch.")

    def setHeadings(singular, plural):
        """Set the heading for singular and plural results."""

    def prevBatchURL():
        """Return a URL to the previous chunk of results."""

    def nextBatchURL():
        """Return a URL to the next chunk of results."""

    def batchPageURLs():
        """Return a list of links representing URLs to pages of
        results."""

class IBatchNavigatorFactory(Interface):
    def __call__(results, request, start=0, size=None, callback=None,
                 transient_parameters=None, force_start=False):
        """Constructs an `IBatchNavigator` instance.

        :param results: is an iterable of results.

        :param request: will be inspected for a start variable; if set,
            it indicates which point we are currently displaying at. It
            will also be inspected for a batch variable; if set, it will
            be used instead of the size supplied in the callsite.  Expected to
            confirm to zope.publisher.interfaces.IRequest.

        :param size: is the default batch size, to fall back to if the
            request does not specify one.  If no size can be determined
            from arguments or request, the launchpad.default_batch_size
            config option is used.

        :param callback: is called, if defined, at the end of object
            construction with the defined batch as determined by the
            start and request parameters.

        :param transient_parameters: optional sequence of parameter
            names that should not be passed on in links generated by
            the batch navigator.  Use this for parameters that had
            meaning when this page was requested, but should not be kept
            around for other page requests in the batch.

        :param force_start: if True, the given `start` argument has
            precedence over the start value in the request, if any.

        :raises InvalidBatchSizeError: if the requested batch size is higher
            than the maximum allowed.
        """
