# Copyright 2009 Canonical Ltd.  All rights reserved.
#
# This file is part of lazr.smtptest
#
# lazr.smtptest is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# lazr.smtptest is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
# License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with lazr.smtptest.  If not, see <http://www.gnu.org/licenses/>.

"""The SMTP test server."""

__metaclass__ = type
__all__ = [
    'QueueServer',
    'Server',
    ]


import smtpd
import socket
import logging
import asyncore

from Queue import Empty
from email import message_from_string


COMMASPACE = ', '
# pylint: disable-msg=C0103
log = logging.getLogger('lazr.smtptest')


class Channel(smtpd.SMTPChannel):
    """A channel that can reset the mailbox."""

    def __init__(self, server, connection, address):
        smtpd.SMTPChannel.__init__(self, server, connection, address)
        self._server = server

    # pylint: disable-msg=W0613
    def smtp_EXIT(self, argument):
        """EXIT - a new SMTP command to cleanly stop the server."""
        self._server.stop()
        self.push('250 Ok')

    def smtp_RSET(self, argument):
        """RSET - hijack this to reset the server instance."""
        self._server.reset()
        smtpd.SMTPChannel.smtp_RSET(self, argument)

    def send(self, data):
        """See `SMTPChannel.send()`."""
        # Call the base class's send method, but catch all socket errors since
        # asynchat/asyncore doesn't do it.
        try:
            return smtpd.SMTPChannel.send(self, data)
        # pylint: disable-msg=W0704
        except socket.error:
            # Nothing here can affect the outcome.
            pass


class Server(smtpd.SMTPServer):
    """An SMTP server."""

    def __init__(self, host, port):
        """Create an SMTP server.

        :param host: The host name to listen on.
        :type host: str
        :param port: The port to listen on.
        :type port: int
        """
        self.host = host
        self.port = port
        smtpd.SMTPServer.__init__(self, (host, port), None)
        self.set_reuse_addr()
        log.info('[SMTPServer] listening: %s:%s', host, port)

    def handle_accept(self):
        """Handle connections by creating our own Channel object."""
        connection, address = self.accept()
        log.info('[SMTPServer] accepted: %s', address)
        Channel(self, connection, address)

    def process_message(self, peer, mailfrom, rcpttos, data):
        """Process a received message."""
        log.info('[SMTPServer] processing: %s, %s, %s, size=%s',
                 peer, mailfrom, rcpttos, len(data))
        message = message_from_string(data)
        message['X-Peer'] = '%s:%s' % (peer[0], peer[1])
        message['X-MailFrom'] = mailfrom
        message['X-RcptTo'] = COMMASPACE.join(rcpttos)
        self.handle_message(message)
        log.info('[SMTPServer] processed message: %s',
                 message.get('message-id', 'n/a'))

    # pylint: disable-msg=R0201
    def start(self):
        """Start the asyncore loop."""
        log.info('[SMTPServer] starting asyncore loop')
        asyncore.loop()

    def stop(self):
        """Stop the asyncore loop."""
        asyncore.socket_map.clear()
        asyncore.close_all()
        self.close()

    # pylint: disable-msg=R0201
    def reset(self):
        """Do whatever you need to do on a reset."""
        log.info('[SMTPServer] reset')

    def handle_message(self, message):
        """Handle the received message.

        :param message: the completed, parsed received email message.
        :type message: `email.message.Message`
        """
        pass


class QueueServer(Server):
    """A server which puts messages in a queue."""

    def __init__(self, host, port, queue):
        """Create an SMTP server which puts messages in a queue.

        :param host: The host name to listen on.
        :type host: str
        :param port: The port to listen on.
        :type port: int
        :param queue: The queue to put messages in.
        :type queue: object with a .put() method taking a single message
            object.
        """
        Server.__init__(self, host, port)
        self.queue = queue

    def handle_message(self, message):
        """See `Server.handle_message()`."""
        self.queue.put(message)

    def reset(self):
        """See `Server.reset()`."""
        while True:
            try:
                self.queue.get_nowait()
            except Empty:
                break
