# Copyright 2010-2015 Canonical Ltd.  This software is licensed under the
# GNU Lesser General Public License version 3 (see the file LICENSE).

"""Events generated by the SSH server."""

__metaclass__ = type
__all__ = [
    'AuthenticationFailed',
    'AvatarEvent',
    'ILoggingEvent',
    'LoggingEvent',
    'ServerStarting',
    'ServerStopped',
    'SFTPClosed',
    'SFTPStarted',
    'UserConnected',
    'UserDisconnected',
    'UserLoggedIn',
    'UserLoggedOut',
    ]

import logging

from zope.interface import (
    Attribute,
    implements,
    Interface,
    )


class ILoggingEvent(Interface):
    """An event is a logging event if it has a message and a severity level.

    Events that provide this interface will be logged in the SSH server access
    log.
    """

    level = Attribute("The level to log the event at.")
    message = Attribute("The message to log.")


class LoggingEvent:
    """An event that can be logged to a Python logger.

    :ivar level: The level to log itself as. This should be defined as a
        class variable in subclasses.
    :ivar template: The format string of the message to log. This should be
        defined as a class variable in subclasses.
    """

    implements(ILoggingEvent)

    def __init__(self, level=None, template=None, **data):
        """Construct a logging event.

        :param level: The level to log the event as. If specified, overrides
            the 'level' class variable.
        :param template: The format string of the message to log. If
            specified, overrides the 'template' class variable.
        :param **data: Information to be logged. Entries will be substituted
            into the template and stored as attributes.
        """
        if level is not None:
            self._level = level
        if template is not None:
            self.template = template
        self._data = data

    @property
    def level(self):
        """See `ILoggingEvent`."""
        return self._level

    @property
    def message(self):
        """See `ILoggingEvent`."""
        return self.template % self._data


class ServerStarting(LoggingEvent):

    level = logging.INFO
    template = '---- Server started ----'


class ServerStopped(LoggingEvent):

    level = logging.INFO
    template = '---- Server stopped ----'


class UserConnected(LoggingEvent):

    level = logging.INFO
    template = '[%(session_id)s] %(address)s connected.'

    def __init__(self, transport, address):
        LoggingEvent.__init__(
            self, session_id=id(transport), address=address)


class AuthenticationFailed(LoggingEvent):

    level = logging.INFO
    template = '[%(session_id)s] failed to authenticate.'

    def __init__(self, transport):
        LoggingEvent.__init__(self, session_id=id(transport))


class UserDisconnected(LoggingEvent):

    level = logging.INFO
    template = '[%(session_id)s] disconnected.'

    def __init__(self, transport):
        LoggingEvent.__init__(self, session_id=id(transport))


class AvatarEvent(LoggingEvent):
    """Base avatar event."""

    level = logging.INFO

    def __init__(self, avatar):
        self.avatar = avatar
        LoggingEvent.__init__(
            self, session_id=id(avatar.transport), username=avatar.username)


class UserLoggedIn(AvatarEvent):

    template = '[%(session_id)s] %(username)s logged in.'


class UserLoggedOut(AvatarEvent):

    template = '[%(session_id)s] %(username)s disconnected.'


class SFTPStarted(AvatarEvent):

    template = '[%(session_id)s] %(username)s started SFTP session.'


class SFTPClosed(AvatarEvent):

    template = '[%(session_id)s] %(username)s closed SFTP session.'
