#!/usr/bin/python
'''
This module ldapdict.py defines classes which wraps an LDAP object in a python dictionary.

The following classes are defined:
    GenericLdapDict      : Readonly class for OpenLDAP Directory, should work with most Directories
    GenericLdapDictWrite : Read / write class for OpenLDAP Directory, should work with most Directories
    ADSLdapDict          : Readonly class for Active Directory
    ADSLdapDictWrite     : Read / write class for Active Directory

For adapting to other Directories like eDirectory the Generic* classes can be easily subclassed.
See ADSLdapDict for example.

$Id: ldapdict.py 1834 2008-03-03 13:27:44Z tongp01 $
$URL: http://10.75.97.97/lnx/iena/aut/trunk/mpauth/mpshared/ldapdict.py $
'''


import sys
import StringIO
import ldap
import ldap.schema
import ldap.modlist
import ldif
from UserDict import DictMixin
# from UserList import UserList
from pprint import pprint as pp
from types import StringType, UnicodeType #, ListType, NoneType
from ldapcache import LdapCache, LdapCacheWrite

__all__ = ['GenericLdapDict', 'GenericLdapDictWrite', 'ADSLdapDict', 'ADSLdapDictWrite', 'runtests']
__version__ = '0.1.0'

class GenericLdapDict(DictMixin, object):
    '''
    Generic LDAP object oriented class represented as a Dictionary object
        GenericLdapDict objects initialized with same LdapCache object
        share information about each other so the same LDAP object will
        only be in memory once. It is possible to initialize multiple
        times. Only the first time a new instance is created.

        All attributes of an GenericLdapDict object can be accessed
        through dictionairy keys. These values are cached.
        The first time the Directory will be accessed.
        Following access through the dictionairy functions
        will return the cached values.

        The GenericLdapDict object also has the following python attributed:
            dn        : distinguished name
            parents   : list of GenericLdapDict objects
                        all parents going up to the root of the directory
                        starting with the direct parent
            children  : list of GenericLdapDict objects
                        all children of the object
            members   : list of GenericLdapDict objects,
                        all members in case of a group object
            memberof  : list of GenericLdapDict objects,
                        all groups the object is a member of


        These attributes also contain cached info. Only at the first
        access they are retrieved from the directory

        The GenericLdapDict objects member functions provide direct
        access to the directory.

        To use the GenericLdapDict object you first have to initialize an LdapCache object
        After that you can use this cache object to create GenericLdapDict instances:
        eg:
        >>> from  ldapcache import LdapCache
        # >>> from ldapdict import GenericLdapDict
        >>> url = "ldap://localhost:389/"
        >>> user = 'cn=manager,dc=example,dc=com'
        >>> passwd = 'secret'
        >>> lc = LdapCache(GenericLdapDict, url, user, passwd,)
        >>> root = lc.root
        >>> bdn = lc.bdn
        >>> manager = GenericLdapDict(user, lc)
        >>> manager.display()

        It is also possible to set a default ldapcache. See GenericLdapDict.setldapcache()

        Changes to the Directory can be made by the GenericLdapDictWrite class.
    '''

    MEMBER_ATTRLIST = ['uniqueMember', 'member']
    MEMBEROF_ATTRLIST = []
    MOD_CLASSDEF = {}

    # Active Directory has extra memberOf attribute
    # and has extra 'hidden' system auxiliary classes ...
    #
    # MEMBEROF_ATTRLIST = ['memberOf',]
    # MOD_CLASSDEF = {'user': ['securityPrincipal', 'mailRecipient'],
    #                'group': ['securityPrincipal', 'mailRecipient']}


    DEFAULT_LDAPCACHE = None

    def setldapcache(cls, ldapcache):
        '''
        Classmethod to set default ldapcache for *LdapDict class
        If used it is not necesary to provide ldapcache in object creation,
        instead this default is used.
        It is still possible to create another ldapcache and use it explicitly
        for other LdapDict instance. for example when connecting two different
        Directory servers in one script.
        eg:
        >>> from  ldapcache import LdapCache
        # >>> from ldapdict import GenericLdapDict
        >>> url = "ldap://localhost:389/"
        >>> user = 'cn=manager,dc=example,dc=com'
        >>> passwd = 'secret'
        >>> lc = LdapCache(GenericLdapDict, url, user, passwd,)
        >>> root = lc.root
        >>> bdn = lc.bdn
        >>> GenericLdapDict.setldapcache(lc)
        >>> manager = GenericLdapDict(user)
        >>> manager.display()

        '''
        cls.DEFAULT_LDAPCACHE = ldapcache
        return ldapcache

    setldapcache = classmethod(setldapcache)

    def __new__(cls, dn , cache = None, data = None):
        '''
        Checks if provided DN is already associated with and GenericLdapDict instance
        in the cache. If so will return associated instance.
        Else will initiate GenericLdapDict object.

        All initialization is done here.

        input:
            dn    : distinguishedName
            cache : LdapCacheObject
            data  : ldapsearch data (don't use this)

        '''
        if cache is None:
            cache = cls.DEFAULT_LDAPCACHE

        dnkey = cache.dnkey(dn)
        if cache.dnidx.has_key(dnkey):
            ol = cache.dnidx[dnkey][0]
        else:
            if data is None:
                try:
                    # dn, data = cache.getdn(dn)
                    res = cache.search(dn, scope = "base")
                    dn, data = res[0]
                except ldap.LDAPError, e:
                    raise e

            ol = object.__new__(cls, dn , cache)
            ol.dn = dn
            ol.data = {}
            ol.parents = []
            ol.cache = cache

            ol.data.update(data)
            cache.dnidx[dnkey] = [ol, ]

            ol.getparents()
            cache.dnidx[dnkey].append(ol.parents[0])

        return ol

    def __init__(self, dn , cache = None, data = None):
        '''
        Dummy __init__
        all initialization is done is __new__
        '''
        pass
#        dnkey = cache.dnkey(dn)
#        if not cache.dnidx.has_key(dnkey):
#            pass

    def __cmp__(self, other):
        '''
        Compares GenericLdapDict object to:
            other GenericLdapDicts
            or string

        used as:
            user = GenericLdapDict('cn=admin,dc=example,dc=com', lc)
            bdn = GenericLdapDict('dc=example,dc=com', lc)
            if user == bnd:
                print "equal"
            else
                print "not equal"

        !!! reverses string because most significant part of dn is on the right !!!
            'cn=admin,dc=example,dc=com'

        If you want to do a plain string compare use .dn attribute or str() like:

            if user.dn > str(bnd)
                print "bigger"
        '''
        if isinstance(other, GenericLdapDict):
            if id(self.cache.dnidx) == id(other.cache.dnidx):
                return self.cache.cmpdn(self.dn, other.dn)
            else:
                res = self.cache.cmpdn(self.dn, other.dn)
                if res == 0:
                    return cmp(self.data, other.data)
                else:
                    return res

        otype = type(other)

        if otype == StringType or otype == UnicodeType:
            try:
                other = self.cache.dnkey(other)[::-1]
            except ldap.DECODING_ERROR:
                other = other.lower()[::-1]

        return cmp(self.cache.dnkey(self.dn)[::-1], other)
        # return cmp(self.__dnkey(self.dn), self.__dnkey(other.dn))

    def __eq__(self, other):
        '''
        Checks if GenericLdapDict object is equal to:
            other GenericLdapDict
            or string
        '''
        return self.__cmp__(other) == 0

    def __ne__(self, other):
        '''
        Checks if GenericLdapDict object is not equal to:
            other GenericLdapDict
            or string
        '''
        return not self == other

    def __str__(self):
        '''
        Displays dn of GenericLdapDict object
        '''
        return self.dn

    def __repr__(self):
        '''
        Displays dn of GenericLdapDict object
        '''
        return self.dn

    def __add__(self, rdn):
        '''
        Add rdn to instance dn and returns GenericLdapDict instance associated with complete dn
        If complete DN does not exist ldap.NO_SUCH_OBJECT is raised.
        eg:
            bdn = GenericLdapDict('dc=example,dc=com', lc)
            user = bdn + 'cn=admin'
            user.display()
        '''
        dn = ','.join([str(rdn), self.dn])
        try:
            ol = self.__class__(dn, self.cache)
        except ldap.NO_SUCH_OBJECT:
            raise ldap.NO_SUCH_OBJECT, "object does not exist %s" % (dn)

        return ol

    def __sub__(self, bdn):
        '''
        Substracts bdn from instance dn and returns rdn string
        If bdn is not parent ArithmeticError is raised.
        eg:
            user = GenericLdapDict('cn=admin,dc=example,dc=com', lc)
            bdn = GenericLdapDict('dc=example,dc=com', lc)
            rdn = user - bdn
            print rdn

            cn=admin
        '''
        try:
            ol = self.__class__(str(bdn), self.cache)
        except ldap.LDAPError:
            raise ArithmeticError, "Invalid dn: %s" % (bdn)

        if ol in self.parents:
            cnt = len(ldap.explode_dn(ol.dn))
            fdn = ldap.explode_dn(self.dn)
            rdn = ','.join([r for r in fdn[:-cnt]])
            return rdn
        else:
            raise ArithmeticError, "Not parent dn: %s" % (bdn)

    def explode_dn(self, dn = None, notypes = False):
        '''
        Explodes DN into list
        '''
        if dn is None:
            dn = self.dn

        return self.cache.explode_dn(str(dn), notypes)

    def keys(self):
        '''
        Returns keys of GenericLdapDict object using class definition
        '''

        keys = []
        classlist = []
        if not self.data.has_key('objectClass'):
            try:
                self.getattr(['objectClass'])
                classlist = self.data['objectClass']
            except KeyError:
                self.getattr(['structuralObjectClass'])
                classlist = self.data['structuralObjectClass']
        else:
            classlist = self.data['objectClass']

        if classlist is not None:
            # In de ADS worden een aantal attributen verstopt ...
#            if ads:
#                if 'user' in classlist or 'group' in classlist:
#                    classlist.append('securityPrincipal')
#                    classlist.append('mailRecipient')
            if self.MOD_CLASSDEF is not None:
                for cls, extra_attrs in self.MOD_CLASSDEF.items():
                    if cls in classlist:
                        classlist.extend(extra_attrs)
            self.cache.getclassdef(classlist)
            for cls in classlist:
                keys.extend(self.cache.classdef[cls]['must'])
                keys.extend(self.cache.classdef[cls]['may'])

            for key in keys:
                if key not in self.data:
                    self.data[key] = None
        else:
            keys = self.data.keys()
        
        # hack around openldap not having RootDSE in schema 
        if 'OpenLDAProotDSE' in classlist:
            keys = self.data.keys()

        # if not hasattr(self,'_keys'):
        #    self._keys = dict([[k.lower(),k] for k in keys])

        return keys

    def _getkeys(self):
        self._keys = dict([[k.lower(),k] for k in self.keys()])
        return self._keys

    def __iter__(self):
        '''
        Yields keys of GenericLdapDict object
        '''
        for k in self.keys():
            yield k

    def has_key(self, key):
        '''
        Checks if GenericLdapDict object has the key
        '''

        return key.lower() in self._keys

    def __getitem__(self, key):
        '''
        Returns value belong to ldap attribute,
            first uses cached value
            else get value from Directory

        used as:
            user = GenericLdapDict('cn=admin,dc=example,dc=com', lc)
            print user['cn']

            admin
        '''
        try:
            key = self._keys[key.lower()]
        except KeyError:
            pass

        if self.data.has_key(key):
            if self.data[key] != None:
                return self.data[key]

        if self.has_key(key):
            self.getattr([key])
        else:
            raise KeyError(key)

        return self.data[key]

    def __getattr__(self, attr):
        '''
        Returns value belong to python attributes:
            children
            members
            memberof
            parents

        eg:
        used as:
            user = GenericLdapDict('cn=admin,dc=example,dc=com', lc)
            print user.members
        '''
        if attr == 'children':
            return self.getchildren()
        elif attr == 'members':
            return self.getmembers()
        elif attr == 'memberof':
            return self.getmemberof()
        elif attr == 'parents':
            return self.getparents()
        elif attr == '_keys':
            return self._getkeys()
        else:
            raise AttributeError(attr)

    def search(self, dn = None, scope = None, classlist = None, attrlist = None, ldapfilter = None):
        '''
        Searches Directory
        input:
            dn           : distinguishedName, if None: instance dn is used
            scope        : ldap search scope; one of: "one", "sub", "base"
            classlist    : list of types of objects to search for
            attrlist     : list of attributes to obtain; if 'None' will get all attributes
            ldapfilter   : plain ldapfilter for advanced searching, will overrule classlist

        result:
            list of GenericLdapDict objects that meets search requirements
        '''

        if dn is None:
            dn = self.dn

        res = self.cache.search(str(dn), scope, classlist, attrlist, ldapfilter)

        dns = []
        for dn, data in res:
            if dn is None:
                continue
            ol = self.__class__(dn, self.cache, data)
            dns.append(ol)
        return dns

    def getattr(self, attrlist = None):
        '''
        Returns LDAP attributes for GenericLdapDict Object
            retrieves attributes directly from Directory
        '''
        if attrlist is not None:
            for key in attrlist:
                self.data[key] = None
        else:
            for key in self.data.keys():
                self.data[key] = None

        data = self.cache.getattr(self.dn, attrlist)

        self.data.update(data)
        self.keys()

    # @classmethod
    def __getparent(cls, dn):
        '''
        Retrieves parent dn from GenericLdapDict object
        '''
        dnlist = ldap.explode_dn(dn)
        parent = ','.join(dnlist[1:])
        return parent
    __getparent = classmethod(__getparent)

    def getparents(self):
        '''
        Retrieves GenericLdapDict objects parents
        '''
        self.parents = []

        parentdn = self.__class__.__getparent(self.dn)

        try:
            parent = self.__class__(parentdn, self.cache)
        except ldap.LDAPError:
            parent = self.cache.dnidx[''][0]
        else:
            self.parents.append(parent)
            parentdn = self.__class__.__getparent(parent.dn)

        # parents.append(self.cache.dnidx[''][0])
        self.parents.extend(parent.parents)

        return self.parents

    def getchildren(self, classtypes = None):
        '''
        Retrieves children of GenericLdapDict object directly from Directory

        output:
            list of GenericLdapDict objects
        '''
        if self.dn == '':
            return self.children

        self.children = []
        if classtypes == None:
            filterstr = '(objectClass=*)'
        elif len(classtypes) == 1:
            filterstr = "(objectClass="+classtypes[0]+")"
        else:
            l = []
            for classtype in classtypes:
                subfilter = "(objectClass="+classtype+")"
                l.append(subfilter)
                filterstr = "(|"+"".join(l)+")"

        self.children = self.search(self.dn, scope = "one", attrlist = ['' ], ldapfilter=filterstr)
        return self.children

    def __getmembers_ads(self):
        '''
        Retrieves members of GenericLdapDict object directly from Directory

        output:
            list of GenericLdapDict objects
        '''
        self.member = []

        if self.has_key('member') and self['member'] is not None:
            for dn in self['member']:
                # ol = self.cache.getdn(dn)
                ol = self.__class__(dn, self.cache)
                self.member.append(ol)

        return self.member

    def __getmemberof_ads(self):
        '''
        Retrieves objects of which GenericLdapDict object is member of directly from Directory

        output:
            list of GenericLdapDict objects
        '''
        self.memberof = []

        if self.has_key('memberOf') and self['memberOf'] is not None:
            for dn in self['memberOf']:
                # ol = self.cache.getdn(dn)
                ol = self.__class__(dn, self.cache)
                self.memberof.append(ol)

        return self.memberof


    def getmembers(self):
        '''
        Retrieves members of GenericLdapDict object directly from Directory
        '''
        members = []

        self.getattr(self.MEMBER_ATTRLIST)
        for memberattr in self.MEMBER_ATTRLIST:
            try:
                members = self[memberattr]
            except:
                pass
            else:
                break
#        try:
#            members = self['uniqueMember']
#        except KeyError:
#            try:
#                members = self['member']
#            except KeyError:
#                return []

        self.members = []
        for member in members:
#            dnkey = self.dnkey(member)
#            if dnkey not in self.cache.dnidx:
#                mo = self.__class__(member, self)
#            else:
#                mo = self.cache.dnidx[dnkey][0]
            mo = self.__class__(member, self.cache)
            # mo.memberof.append(self)
            self.members.append(mo)

        return self.members

    def getmemberof(self):
        '''
        Retrieves object of which GenericLdapDict object is member of directly from Directory
        '''
        self.memberof = []

        if len(self.MEMBEROF_ATTRLIST) > 0:
            self.getattr(self.MEMBEROF_ATTRLIST)
            for attr in self.MEMBEROF_ATTRLIST:
                if self.has_key(attr) and self[attr] is not None:
                    for dn in self[attr]:
                        # ol = self.cache.getdn(dn)
                        ol = self.__class__(dn, self.cache)
                        self.memberof.append(ol)
        else:
            if len(self.MEMBER_ATTRLIST) == 1:
                ldapfilter = self.MEMBER_ATTRLIST[0] + "=" + self.dn
            else:
                ldapfilterlist = ["(|"]
                for memberattr in self.MEMBER_ATTRLIST:
                    f = "(%s=%s)" %(memberattr, self.dn)
                    ldapfilterlist.append(f)
                ldapfilterlist.append(")")

                ldapfilter = ''.join(ldapfilterlist)

            bdn = self.cache.bdn.dn
            self.memberof = self.search(dn = bdn, scope = 'sub', attrlist = ['objectClass'], ldapfilter=ldapfilter)

        return self.memberof

    def gettree(self, classtypes = None):
        '''
        Retrieves recursivelly all children of GenericLdapDict object.
        A good way to fill the cache
        '''
        if classtypes is not None:
            if 'organizationalUnit' not in classtypes:
                classtypes.append('organizationalUnit')
                self.getchildren(classtypes)

        if len(self.children) != 0:
            for child in self.children:
                if child['objectClass'] is not None:
                    if 'organizationalUnit' in child['objectClass']:
                        child.gettree(classtypes)

        return True

    def printtree_simple(self, indent = 3, level = 0):
        '''
        prints objects sub tree
        '''
        indent = int(indent)
        level = int(level)

        dn = self.dn
        try:
            rdn = ldap.explode_dn(dn)[0]
        except IndexError:
            rdn = '[rootDSE]'

        if level == 0:
            if dn != '':
                rdn = dn

        objcl = " : (" + sorted(self['objectClass'])[0] + ")"

        fill = indent * level * ' '
        print fill + rdn + objcl
        level += 1

        if len(self.children) != 0:
            for child in sorted(self.children):
                child.printtree_simple(indent, level)

        return level

    def printtree(self, indent = 3, level = 0, fill = None, last = True ):
        '''
        prints objects sub tree with lines
        '''
        indent = int(indent)
        level = int(level)

        dn = self.dn
        try:
            rdn = ldap.explode_dn(dn)[0]
        except IndexError:
            rdn = '[rootDSE]'

        if level == 0:
            if dn != '':
                rdn = dn
            fill = []

        if level == 0:
            print
            s = '>> '
        elif last:
            s = '+- '
        else:
            s = '|- '

        if self['objectClass'] is not None:
            # objcl = " : (" + sorted(self['objectClass'])[0] + ")"
            objcl = " : (" + self['objectClass'][-1] + ")"
        else:
            objcl = " : (None)"
        print ''.join(fill) + s + rdn + objcl
        fill.append('|  ')
        level += 1

        if level > 0 and last:
            fill[-1] = '   '

        children = sorted(self.children)
        if len(children) > 1:
            for child in children[:-1]:
                fill = child.printtree(indent, level, fill, False)

        for child in children[-1:]:
            fill = child.printtree(indent, level, fill, True)
            print ''.join(fill)

        if len(fill) > 0:
            del fill[-1]

        return fill

    def display(self, all = False):
        '''
        Displays GenericLdapDict object
        '''
        l = []
        l.append(str(self).center(80, '-'))
        # for key, value in self.data.items():

        if all:
            for key, value in sorted(self.items()):
                l.append( "%35s : %r" % (key, value))
        else:
            for key, value in sorted(self.data.items()):
                if value is not None:
                    l.append( "%35s : %r" % (key, value))

        l.append('')
        l.append("%35s : %r" % ("self.children", [str(ol) for ol in self.children]))
        l.append("%35s : %r" % ("self.parents", [str(ol) for ol in self.parents]))
        l.append("%35s : %r" % ("self.members", [str(ol) for ol in self.members]))
        l.append("%35s : %r" % ("self.memberof", [str(ol) for ol in self.memberof]))
        l.append("-"*80)

        return '\n'.join(l)

    def toLdif(self):
        '''
        Returns LDIF for GenericLdapDict object
        '''
        output = StringIO.StringIO()

        toldif = ldif.LDIFWriter(output)

        # skip the 'None' values
        data = {}
        for key, value in self.data.items():
            if value is not None:
                data[key] = value

        toldif.unparse(self.dn, data)
        contents = output.getvalue()
        output.close()

        return contents

class GenericLdapDictWrite(GenericLdapDict):
    '''
    Extends GenericLdapDict with methods to change objects in Directory.
    Ldapcache needs to be of LdapCacheWrite type,
    otherwise TypeError is raised!!!
    '''
    def __new__(cls, dn , cache = None, data = None):
        if cache is None:
            cache = cls.DEFAULT_LDAPCACHE

        if isinstance(cache, LdapCacheWrite):
            return GenericLdapDict.__new__(cls, dn , cache, data)
        else:
            raise TypeError, "Wrong cache type: need \'LdapCacheWrite\'"

    def __setitem__(self, key, value):
        '''
        Sets / updates GenericLdap object key with value.
        Use commit to actually update the attribute in de Directory
        '''
        if not hasattr(self, "__olddata"):
            self.__olddata = {}

        key = self._keys[key.lower()]
        self.__olddata[key] = self[key]
        self.data[key] = value

#    def __del__(self):
#        pass

    def __setattr(self, attrdict):
        '''
        Sets / updates LDAP object attribute with value.
        '''
        modlist = []

        attrlist = attrdict.keys()
        current = self.cache.getattr(self.dn, attrlist)

        for key, value in attrdict.items():
            if attrdict[key] != current.get(key, None):
                if value == None:
                    modlist.append((ldap.MOD_DELETE, key, value))
                elif self[key] is not None:
                    modlist.append((ldap.MOD_REPLACE, key, value))
                else:
                    modlist.append((ldap.MOD_ADD, key, value))

        try:
            self.cache.setattr(self.dn, modlist)
        except ldap.LDAPError:
            raise

        res = self.cache.getattr(self.dn, attrlist)

        # self.data.update(attrdict)
        # next statement does not work on delete!!!
        self.data.update(res)

    def setattr(self, attrdict , force = False):
        '''
            sets / updates LDAP object attribute with value.
        '''
        attrlist = attrdict.keys()
        current = self.cache.getattr(self.dn, attrlist)

        for key in attrlist:
            if self[key] != None and current[key] != self[key] and not force:
                errormsg = "Dirty attribute: \'" + key +"\'"
                raise RuntimeError(errormsg)

        self.__setattr(attrdict)

    def commit(self, force = False):
        '''
        Commits changes from GenericLdap object to corresponding LDAP object in Directory.
        If force = False and attribute value has changed in the mean while
        RuntimeError error is raised.
        '''
        attrlist = self.__olddata.keys()
        current = self.cache.getattr(self.dn, attrlist)

        attrdict = {}
        for key in attrlist:
            if self.__olddata[key] != None and current[key] != self.__olddata[key] and not force:
                errormsg = "Dirty attribute: \'" + key +"\'"
                raise RuntimeError(errormsg)

            attrdict[key] = self.data[key]

        self.__olddata.clear()
        self.__setattr(attrdict)

#    def adddn(self, dn, modlist):
#        dn = self.cache.adddn(self, dn, modlist)
#        return self.__class__(dn, self.cache)


    def adddn(self, dn, classlist, attrdict):
        '''
        Adds DN to Directory
        '''
        try:
            ol = self.__class__(dn, self.cache)
        except ldap.NO_SUCH_OBJECT:
            pass
        else:
            raise ldap.ALREADY_EXISTS, dn

        must = []
        for cls in classlist:
            if cls not in self.cache.classdef:
                self.cache.getclassdef([cls])
            must.extend(self.cache.classdef[cls]['must'])

        for key in must:
            if key not in attrdict:
                errormsg = "Compulsory key \'%s\' not provided" % (key)
                raise KeyError(errormsg)

        modlist = []

        # attrlist = attrdict.keys()
        modlist.append(('objectClass' , classlist))
        for key, value in attrdict.items():
            if value is not None and key != 'objectClass':
                modlist.append((key, value))

#        modlist = [('objectClass', ['groupOfUniqueNames', 'top']),
#                   ('uniqueMember', ['cn=tongp01,ou=UID,dc=example,dc=com']),
#                   ('cn', ['Host2'])]

        dn = self.cache.adddn(dn, modlist)
        ol = self.__class__(dn, self.cache)
        return ol


    def addchild(self, rdn, classlist, attrdict):
        dn = ','.join([rdn, self.dn])
        ol = self.adddn(dn, classlist, attrdict)
        return ol

    def deldn(self):
        '''
        Delets DN from Directory
        '''
        dnkey = self.cache.dnkey(self.dn)
        if self in self.parents[0].children:
            self.parents[0].children.remove(self)
        for member in self.members:
            if self in member.memberof:
                member.memberof.remove(self)
        for mof in self.memberof:
            if self in mof.members:
                mof.members.remove(self)

        self.cache.deldn(self.dn)

        del self.cache.dnidx[dnkey]

class ADSLdapDict(GenericLdapDict):
    '''
    Class to connect to Active Directory.
    It has some different attributes and has an extra systemAuxiliaryClass
    which adds some extra attributes that are not seen by the standard schema.
    MOD_CLASSDEF is used for this.
    See GenericLdapDict for more info.

    Changes to the Active Directory can be made by the ADSLdapDictWrite class.
    '''
    MEMBER_ATTRLIST = ['member']
    MEMBEROF_ATTRLIST = ['memberOf']
    MOD_CLASSDEF = {'user': ['securityPrincipal', 'mailRecipient'],
                    'group': ['securityPrincipal', 'mailRecipient']}

class ADSLdapDictWrite(ADSLdapDict, GenericLdapDictWrite):
    '''
    Class to change objects in Active Directory.
    See ADSLdapDict and GenericLdapDictWrite for more info.

    If ADSLdapCacheWrite is used as ldapcache, passwords can be changed.
    '''
    pass



def _test():
    '''
    Execute doctest
    '''
    import doctest
    doctest.testmod(verbose = True)

def runtests():
    import ldaptest
    try:
        ldaptest.runtests_ol()
    except:
        pass
    
    try:
        ldaptest.runtests_ads()
    except:
        pass
    
if __name__ == "__main__":
    # _test()
    runtests()



