
'''
mrucache:

Dictionary type object which caches a certain amount of objects. 
An object can be retrieved from the cache using a unique identifier, 
anything that can be a dictionary key. 

This cache can be used for caching network resources like LDAP result sets.
   
If an object is pushed out of the cache and it is still reference anywhere,
it will still be available from the cache, but though weak reference.  
As soon as the object is not references anymore, 
it will be garbage collected and the weak reference is gone.
  
'''

import gc
from weakref import WeakValueDictionary, getweakrefcount, getweakrefs

class MRUCache(WeakValueDictionary, object):
    def __init__(self, size=20, debug= False):
        self.size = size
        self.__debug = debug
        self.mrulist = []
        WeakValueDictionary.__init__(self)

    def __getitem__(self, key):
#        data = None
#        if self.has_key(key):
#            data = self.get(key)
#        else:
#            raise KeyError(key)
        data = WeakValueDictionary.__getitem__(self, key)
        
        self.updatemru(data)
        return data

    def __setitem__(self, key, value):
        WeakValueDictionary.__setitem__(self, key, value)
        self.updatemru(value)

    def updatemru(self, value):
        #if it is already on top, do not update mrulist
        first = False
        try:
            first = self.mrulist[0] == value
        except IndexError:
            first = False
            
        if not first:  
            try:
                self.mrulist.remove(value)
            except ValueError:
                pass
    
            self.mrulist.insert(0,value)
        
        if len(self.mrulist) > self.size:
            del self.mrulist[self.size:]
            
    def weakrefcount(self):        
        return len(self) - len(self.mrulist) 

    def has_weakrefs(self):        
        return self.weakrefcount() != 0
    
    def clear(self, value = None):
        if value is None:
            self.mrulist = []
        else:
            try:
                self.mrulist.remove(value)
            except ValueError:
                pass            

    def __str__(self):
        l = []
        for k, v in self.iteritems():
            l.append("%s: '%s'" % (k, v) )

        return "{%s}" % (', '.join(l))
class C:
    nrinst = 0
    def __init__(self):
        self.nr = self.nrinst
        C.nrinst = C.nrinst + 1

    def __repr__(self):
        return "nr: %s" % (self.nr)

def main():
    cache = MRUCache(size = 2, debug = True)

    nr = 16
    cs = []
    for i in range(nr):
        c = C()
        cs.append(c)
        cache[i] = c

    print "cs: ", cs

    a = cache[9]
    b = cache[11]
    d = cache[12]

    del cs[6:]

    print "cs: ", cs
    print "mrulist %s: %s" % (len(cache.mrulist), cache.mrulist)
    print "weak %s: %s" % (len(cache), cache)
    print cache.weakrefcount()
    #print cache

      
    print id(cs)
    for i in range(nr):
        try:
            c = cache[i]
            print "%s, %s, %s" % (c, getweakrefcount(c), id(gc.get_referrers(c)[1]))

        except KeyError:
            print "Removed from cache: %s" % (i)

    for i in [1,2,4,5]:
        try:
            c = cache[i]
            print "%s, %s, %s" % (c, getweakrefcount(c), id(gc.get_referrers(c)[1]))

        except KeyError:
            print "Removed from cache: %s" % (i)

    print "cs: ", cs
    print "mrulist %s: %s" % (len(cache.mrulist), cache.mrulist)
    print "weak %s: %s" % (len(cache), cache)
    print cache.weakrefcount()

    del a
    del cs
    del b
    a = cache[12]
    print id(a), id(d)
    
    cache.updatemru(d)
    # print "cs: ", cs
    print "mrulist %s: %s" % (len(cache.mrulist), cache.mrulist)
    print "weak %s: %s" % (len(cache), cache)
    print cache.weakrefcount()    

if __name__ == "__main__":
    main()
