import ldap
import logging

from ldapper.messages import LdapSearchResult


logger = logging.getLogger(__name__)


class LdapClient(object):

    LDAP_VERSION = None

    def __init__(self, host, *args, **kwargs):
        self.host = host
        self.basedn = kwargs.pop('basedn', None)
        self.LDAP_VERSION = kwargs.pop('version', ldap.VERSION3)

    def bind(self, username=None, password=None):
        self.ldap = ldap.initialize(self.host)
        logger.debug('Initialized ldap to %s' % self.host)
        self.ldap.protocol_version = self.LDAP_VERSION
        logger.debug('Using LDAP version %s' % self.LDAP_VERSION)

        if username and password:
            logger.debug('Binding to ldap as user %s' % username)
            self.ldap.simple_bind_s(username, password)

    def unbind(self):
        logger.debug('Unbinding from %s' % self.host)
        self.ldap.unbind_s()

    def search(self, filter=None, props=['*'], subtree_dn=None):
        if subtree_dn:
            basedn = ','.join([subtree_dn,self.basedn])
        else:
            basedn = self.basedn

        logger.debug('Searching the tree %s' % basedn)

        result = self.ldap.search_s(basedn,
                                    ldap.SCOPE_SUBTREE,
                                    filter,
                                    props)
        logger.debug('Found %d results' % len(result))
        return [LdapSearchResult(item) for item in result]

    def change_password_for(self, dn, old_pass, new_pass):
        logger.debug('Changing password for dn %s' % dn)
        result = self.ldap.passwd_s(dn, old_pass, new_pass)
        return result

