# Copyright 2014 NYBX Inc.
# All rights reserved.

"""
:module: ledgerx.api.client.test.test_client_protocol
:synopsis: Unit tests for client protocol messages.
:author: Amr Ali <amr@ledgerx.com>

This test module is only concerned with version-agnostic messages.
See <test_client_protocol_x_x_x> modules for specific message versions.
"""

import unittest

from semantic_version import Version

from ledgerx.api.client.protocol import (
        ClientMessage, MessageStatus, MessageParser)

class TestClientMessage(unittest.TestCase):

    def test_finalize(self):
        msg = ClientMessage()
        t = msg.timestamp
        msg.dumps()
        self.assertNotEqual(t, msg.timestamp)

    def test_client_finalize(self):
        class __TestMsg(ClientMessage):
            def client_finalize(self):
                self._mversion = '0'
                self.test = 'test'
        msg = __TestMsg()
        msg.dumps()
        self.assertEqual(msg.test, 'test')

    def test_client_reply(self):
        msg = ClientMessage()
        rep = msg.reply()
        self.assertIsInstance(rep, MessageStatus)
        self.assertEqual(rep.mversion, msg.mversion)
        self.assertTrue(msg.fullfills(ClientMessage))

class TestMessageStatus(unittest.TestCase):

    def test_status_codes(self):
        self.assertEqual(MessageStatus.STATUS_OK, 200)
        self.assertEqual(MessageStatus.STATUS_ACK, 300)
        self.assertEqual(MessageStatus.STATUS_CLIENT_ERROR, 400)
        self.assertEqual(MessageStatus.STATUS_CLIENT_DENIED, 401)
        self.assertEqual(MessageStatus.STATUS_SERVER_ERROR, 500)

    def test_ok(self):
        msg = MessageStatus().ok('test')
        self.assertEqual(msg.status, MessageStatus.STATUS_OK)
        self.assertEqual(msg.message, 'test')

    def test_ack(self):
        msg = MessageStatus().ack('test')
        self.assertEqual(msg.status, MessageStatus.STATUS_ACK)
        self.assertEqual(msg.message, 'test')

    def test_client_error(self):
        msg = MessageStatus().client_error('test')
        self.assertEqual(msg.status, MessageStatus.STATUS_CLIENT_ERROR)
        self.assertEqual(msg.message, 'test')

    def test_access_denied(self):
        msg = MessageStatus().access_denied('test')
        self.assertEqual(msg.status, MessageStatus.STATUS_CLIENT_DENIED)
        self.assertEqual(msg.message, 'test')

    def test_server_error(self):
        msg = MessageStatus().server_error('test')
        self.assertEqual(msg.status, MessageStatus.STATUS_SERVER_ERROR)
        self.assertEqual(msg.message, 'test')

    def test_set(self):
        msg = MessageStatus().set(MessageStatus.STATUS_ACK, 'test', 'data')
        self.assertEqual(msg.status, MessageStatus.STATUS_ACK)
        self.assertEqual(msg.message, 'test')
        self.assertEqual(msg.data, 'data')

class TestMessageParser(unittest.TestCase):

    def test_parse(self):
        msg = ClientMessage()
        res = MessageParser.parse(msg.dumps())
        self.assertIsInstance(res, MessageStatus)
        self.assertEqual(res.status, MessageStatus.STATUS_CLIENT_ERROR)
        self.assertIn('version', res.message)

    def test_oldest_newest(self):
        omodule = MessageParser.oldest_version_module()
        self.assertEqual(omodule.ClientMessage.Version, '0.0.1')

        nmodule = MessageParser.newest_version_module()
        self.assertTrue(
                Version(nmodule.ClientMessage.Version) >=
                Version(omodule.ClientMessage.Version))

