# Copyright 2014 NYBX Inc.
# All rights reserved.

"""
:module: ledgerx.api.client.protocol
:synopsis: Client protocol message structures.
:author: Amr Ali <amr@ledgerx.com>
"""

from semantic_version import Version

from ledgerx.protocol.utils import import_versions
from ledgerx.protocol.messages import (
        MsgPackMessage, MessageIDMixin, MessageVersionMixin,
        MessageTypeMixin, MessageTimeMixin, BaseMessageStatus,
        BaseMessageParser, MessageMeta, MessageField)

class MessageStatus(
        BaseMessageStatus, MessageIDMixin, MessageVersionMixin,
        MessageTypeMixin, MsgPackMessage):
    """\
    A status message to report the status of a previous message.
    """
    Type = 'status'
    STATUS_OK = 200
    STATUS_ACK = 300
    STATUS_CLIENT_ERROR = 400
    STATUS_CLIENT_DENIED = 401
    STATUS_SERVER_ERROR = 500

    def ok(self, message=None):
        return self.set(self.STATUS_OK, message)

    def ack(self, message=None):
        return self.set(self.STATUS_ACK, message)

    def client_error(self, message):
        return self.set(self.STATUS_CLIENT_ERROR, message)

    def access_denied(self, message):
        return self.set(self.STATUS_CLIENT_DENIED, message)

    def server_error(self, message):
        return self.set(self.STATUS_SERVER_ERROR, message)

    def set(self, code, message, data=None):
        self.status = code
        self.message = message
        self.data = data
        return self

class ClientMessage(
        MsgPackMessage, MessageIDMixin, MessageVersionMixin,
        MessageTypeMixin, MessageTimeMixin):
    """\
    Base abstract message for all client protocol messages.
    """
    MessageStatus = MessageStatus

    def finalize(self):
        self.refresh_timers()
        self.client_finalize()

    def reply(self):
        """\
        Return a message status with the same message ID.
        """
        smsg = self.MessageStatus()
        smsg.mid = self.mid
        smsg.mversion = self.mversion
        return smsg

    def client_finalize(self):
        """\
        Override this method to run a finalizer before message serialization.
        """
        pass

class MessageParser(BaseMessageParser):
    """\
    Message parser for client messages.
    """
    ParentMessage = ClientMessage
    MessageStatus = MessageStatus
    MessageVersions = import_versions(__file__, 'ledgerx.api.client')

    @classmethod
    def newest_version_module(cls):
        """\
        Return the newest interface version module.
        """
        return cls.MessageVersions[cls.__get_x_version(max)]

    @classmethod
    def oldest_version_module(cls):
        """\
        Return the oldest interface version module.
        """
        return cls.MessageVersions[cls.__get_x_version(min)]

    @classmethod
    def __get_x_version(cls, functor):
        return str(functor(
            map(lambda x: Version(x), cls.MessageVersions.keys())
            ))

