# Copyright 2014 NYBX Inc.
# All rights reserved.

"""
:module: ledgerx.api.client.v0.0.1
:synopsis: Client protocol message structures.
:author: Amr Ali <amr@ledgerx.com>

Client protocol messages v0.0.1
"""

import re

from ledgerx.protocol.messages import MessageMeta, MessageField, record_message_type
from ledgerx.api.client.protocol import ClientMessage, MessageStatus

UUID_HEX_NUM_BYTES = 32

@record_message_type
class MessageStatus(MessageStatus):
    """\
    A specialization of :class:`ledgerx.api.client.protocol.MessageStatus` to account for
    status messages that are more related to this particular version.
    """
    Version = '0.0.1'
    STATUS_CONTRACT_NOT_FOUND = 600
    STATUS_ENTRY_NOT_FOUND = 601
    STATUS_ENTRYID_IS_INVALID = 602
    STATUS_CANCEL_FAILURE = 603
    STATUS_CANCEL_REPLACE_FAILURE = 604
    STATUS_REDUCE_SIZE_UNDERFLOW = 605
    STATUS_REDUCE_SIZE_FAILURE = 606
    STATUS_MARKET_HOURS_DENIED = 607
    STATUS_INVALID_MIN_INCREMENT = 608
    STATUS_POST_DENIED = 609
    STATUS_CANCEL_SUCCESS = 700
    STATUS_CANCEL_REPLACE_SUCCESS = 701
    STATUS_REDUCE_SIZE_SUCCESS = 702
    STATUS_UNAUTHORIZED = 800

class ClientMessage(ClientMessage):
    """\
    A specialization of :class:`ledgerx.api.client.protocol.ClientMessage` to use the
    specialized MessageStatus for this particular protocol version.
    """
    Version = '0.0.1'
    MessageStatus = MessageStatus

class MessageContractIDMixin(object, metaclass=MessageMeta):

    @MessageField
    def contract_id(self):
        if not self._contract_id:
            self._contract_id = 0
        return self._contract_id

    @contract_id.setter
    def contract_id(self, val):
        self._contract_id = int(val)

class MessageEntryIDMixin(object, metaclass=MessageMeta):

    @MessageField
    def entry_id(self):
        if self._entry_id is None:
            self._entry_id = b''.rjust(UUID_HEX_NUM_BYTES, b'0')
        return self._entry_id

    @entry_id.setter
    def entry_id(self, val):
        if isinstance(val, str):
            val = bytes(val, 'utf8')
        self._entry_id = bytes(val).rjust(UUID_HEX_NUM_BYTES, b'0')

class MessageTIMixin(object, metaclass=MessageMeta):
    _pattern = re.compile(r"[a-zA-Z0-9]+([_:.-][a-zA-Z0-9]+)*")

    @MessageField
    def ti(self):
        return self._ti

    @ti.setter
    def ti(self, val):
        if not self._pattern.search(val):
            raise ValueError("invalid transaction identifier")
        self._ti = val

class MessageSubmissionMixin(MessageContractIDMixin):
    """\
    A message mixin to describe a submission entry for other messages.
    """

    @MessageField
    def is_ask(self):
        if not self._is_ask:
            self._is_ask = False
        return self._is_ask

    @is_ask.setter
    def is_ask(self, val):
        self._is_ask = bool(val)

    @MessageField
    def size(self):
        if not self._size:
            self._size = 0
        return self._size

    @size.setter
    def size(self, val):
        val = int(val)
        if not val > 0:
            raise ValueError("size must be positive")
        self._size = val

    @MessageField
    def price_in_cents(self):
        if not self._price_in_cents:
            self._price_in_cents = 0
        return self._price_in_cents

    @price_in_cents.setter
    def price_in_cents(self, val):
        val = int(val)
        if not val > 0:
            raise ValueError("price_in_cents must be positive")
        self._price_in_cents = val

    @property
    def mox2(self):
        """\
        A placeholder message field to be implemented by the appropriate
        subclass.
        """
        return False

    @property
    def scp_sweeper_mpid(self):
        """\
        A placeholder message field to be implemented by the appropriate
        subclass.
        """
        return 0

    @property
    def scp_sweeper_cid(self):
        """\
        A placeholder message field to be implemented by the appropriate
        subclass.
        """
        return 0

class MessageActionReportMixin(MessageContractIDMixin):
    """\
    A message mixin to describe an AR like message.
    """
    STATUS_CROSS_DENIED = 100
    STATUS_INSERTED = 200
    STATUS_FILLED = 201
    STATUS_NOFILL = 202
    STATUS_CANCELED = 203

    REASON_ERROR = 50
    REASON_NOT_PERMITTED = 51

    @MessageField
    def inserted_price(self):
        if not self._inserted_price:
            self._inserted_price = 0
        return self._inserted_price

    @inserted_price.setter
    def inserted_price(self, val):
        self._inserted_price = int(val)

    @MessageField
    def inserted_size(self):
        if not self._inserted_size:
            self._inserted_size = 0
        return self._inserted_size

    @inserted_size.setter
    def inserted_size(self, val):
        self._inserted_size = int(val)

    @MessageField
    def filled_price(self):
        if not self._filled_price:
            self._filled_price = 0
        return self._filled_price

    @filled_price.setter
    def filled_price(self, val):
        self._filled_price = int(val)

    @MessageField
    def filled_size(self):
        if not self._filled_size:
            self._filled_size = 0
        return self._filled_size

    @filled_size.setter
    def filled_size(self, val):
        self._filled_size = int(val)

    @MessageField
    def status_type(self):
        if not self._status_type:
            self._status_type = 0
        return self._status_type

    @status_type.setter
    def status_type(self, val):
        self._status_type = int(val)

    @MessageField
    def status_reason(self):
        if not self._status_reason:
            self._status_reason = 0
        return self._status_reason

    @status_reason.setter
    def status_reason(self, val):
        self._status_reason = int(val)

    @MessageField
    def is_ask(self):
        if not self._is_ask:
            self._is_ask = False
        return self._is_ask

    @is_ask.setter
    def is_ask(self, val):
        self._is_ask = bool(val)

    @MessageField
    def clock(self):
        if not self._clock:
            self._clock = 0
        return self._clock

    @clock.setter
    def clock(self, val):
        self._clock = int(val)

@record_message_type
class MessageLimitOrder(ClientMessage, MessageSubmissionMixin):
    """\
    A message for posting a customer limit order.
    """
    Type = 'customer_limit_order'

@record_message_type
class MessageSCPOrder(ClientMessage, MessageSubmissionMixin):
    """\
    A message for posting a SCP order.
    """
    Type = 'scp_sweeper_order'

    @MessageField
    def scp_sweeper_mpid(self):
        if not self._scp_sweeper_mpid:
            self._scp_sweeper_mpid = None
        return self._scp_sweeper_mpid

    @scp_sweeper_mpid.setter
    def scp_sweeper_mpid(self, val):
        self._scp_sweeper_mpid = int(val)

    @MessageField
    def scp_sweeper_cid(self):
        if not self._scp_sweeper_cid:
            self._scp_sweeper_cid = None
        return self._scp_sweeper_cid

    @scp_sweeper_cid.setter
    def scp_sweeper_cid(self, val):
        self._scp_sweeper_cid = int(val)

    @MessageField
    def scp_conversation_id(self):
        if not self._scp_conversation_id:
            self._scp_conversation_id = None
        return self._scp_conversation_id

    @scp_conversation_id.setter
    def scp_conversation_id(self, val):
        self._scp_conversation_id = int(val)

@record_message_type
class MessageMarketOrder(ClientMessage, MessageSubmissionMixin):
    """\
    A message for posting a customer market order.
    """
    Type = 'customer_market_order'

@record_message_type
class MessageQuote(ClientMessage, MessageSubmissionMixin):
    """\
    A message for posting a market maker quote.
    """
    Type = 'market_maker_quote'

    @MessageField
    def mox2(self):
        if not self._mox2:
            self._mox2 = False
        return self._mox2

    @mox2.setter
    def mox2(self, val):
        self._mox2 = bool(val)

@record_message_type
class MessageCancelReplace(
        ClientMessage, MessageEntryIDMixin, MessageSubmissionMixin):
    """\
    A message to atomically swap a new entry for an outstanding entry.
    """
    Type = 'cancel_replace_entry'

@record_message_type
class MessageCancel(
        ClientMessage, MessageContractIDMixin, MessageEntryIDMixin):
    """\
    A message to cancel an order.
    """
    Type = 'cancel_entry'

@record_message_type
class MessageCancelAll(
        ClientMessage):
    """\
    A message to cancel all orders across all contracts.
    """
    Type = 'cancel_all'

@record_message_type
class MessageReduceSize(ClientMessage, MessageContractIDMixin,
        MessageEntryIDMixin):
    """\
    A message to reduce the size of an entry.
    """
    Type = 'reduce_entry_size'

    @MessageField
    def size_decrement(self):
        if not self._size_decrement:
            self._size_decrement = 0
        return self._size_decrement

    @size_decrement.setter
    def size_decrement(self, val):
        val = int(val)
        if not val > 0:
            raise ValueError("size_decrement must be positive")
        self._size_decrement = val

@record_message_type
class MessageActionReport(ClientMessage, MessageActionReportMixin):
    """\
    An action report message.
    """
    Type = 'action_report'

@record_message_type
class MessageGetBookState(ClientMessage, MessageContractIDMixin):
    """\
    A message to request a book state.
    """
    Type = 'get_book_state'

@record_message_type
class MessageBookState(ClientMessage, MessageContractIDMixin, MessageEntryIDMixin):
    """\
    A message to describe a book state.
    """
    Type = 'book_state'

    @MessageField
    def is_ask(self):
        if not self._is_ask:
            self._is_ask = False
        return self._is_ask

    @is_ask.setter
    def is_ask(self, val):
        self._is_ask = bool(val)

    @MessageField
    def price(self):
        if not self._price:
            self._price = 0
        return self._price

    @price.setter
    def price(self, val):
        self._price = int(val)

    @MessageField
    def size(self):
        if not self._size:
            self._size = 0
        return self._size

    @size.setter
    def size(self, val):
        self._size = int(val)

    @MessageField
    def is_end(self):
        if not self._is_end:
            self._is_end = False
        return self._is_end

    @is_end.setter
    def is_end(self, val):
        self._is_end = bool(val)

    @MessageField
    def clock(self):
        if not self._clock:
            self._clock = 0
        return self._clock

    @clock.setter
    def clock(self, val):
        self._clock = int(val)

@record_message_type
class MessageBookTop(ClientMessage, MessageContractIDMixin):
    """\
    A message to report the top of the book to clients.
    """
    Type = 'book_top'

    @MessageField
    def ask(self):
        if not self._ask:
            self._ask = 0
        return self._ask

    @ask.setter
    def ask(self, val):
        self._ask = int(val)

    @MessageField
    def bid(self):
        if not self._bid:
            self._bid = 0
        return self._bid

    @bid.setter
    def bid(self, val):
        self._bid = int(val)

@record_message_type
class MessageGetContractDetail(ClientMessage, MessageContractIDMixin):
    """\
    A message to request details for a particular contract ID.
    """
    Type = 'get_contract_detail'

    @MessageField
    def all_contracts(self):
        if not self._all_contracts:
            self._all_contracts = False
        return self._all_contracts

    @all_contracts.setter
    def all_contracts(self, val):
        self._all_contracts = bool(val)

    def contract_detail(self):
        m = MessageContractDetail()
        m.mid = self.mid
        return m

@record_message_type
class MessageContractDetail(ClientMessage, MessageContractIDMixin):
    """\
    A message that describes the details of a contract.
    """
    Type = 'contract_detail'

    CONTRACT_TYPE_PUT = 100
    CONTRACT_TYPE_CALL = 101

    @MessageField
    def expiration(self):
        if not self._expiration:
            self._expiration = 0
        return self._expiration

    @expiration.setter
    def expiration(self, val):
        self._expiration = int(val)

    @MessageField
    def strike_price(self):
        if not self._strike_price:
            self._strike_price = 0
        return self._strike_price

    @strike_price.setter
    def strike_price(self, val):
        self._strike_price = int(val)

    @MessageField
    def contract_type(self):
        return self._contract_type

    @contract_type.setter
    def contract_type(self, val):
        if val not in [self.CONTRACT_TYPE_CALL, self.CONTRACT_TYPE_PUT]:
            raise ValueError("contract type must be either a put or a call")
        self._contract_type = int(val)

@record_message_type
class MessageExerciseRequest(ClientMessage, MessageTIMixin):
    """\
    A message to request exercising an option.
    """
    Type = 'exercise_request'

@record_message_type
class MessageReplay(ClientMessage):
    """\
    A message to request a replay of all ARs.
    """
    Type = 'replay'

    @MessageField
    def begin_timestamp(self):
        return self._begin_timestamp

    @begin_timestamp.setter
    def begin_timestamp(self, val):
        if not isinstance(val, int):
            raise ValueError("timestamp must be of type int")
        self._begin_timestamp = val

    @MessageField
    def end_timestamp(self):
        return self._end_timestamp

    @end_timestamp.setter
    def end_timestamp(self, val):
        if not isinstance(val, int):
            raise ValueError("timestamp must be of type int")
        self._end_timestamp = val

