class HTTPError(Exception):
    def __init__(self, code, message):
        self.code = code
        self.message = message

    def __str__(self):
        return "\nResponse code: %s\nResponse: %s" % (self.code, self.message)

#region Default

import httplib
import socket


class Connection(object):
    def __init__(self, port=6521, host="127.0.0.1", retry=1):
        self.url = "http://%s:%s/%%s" % (host, port)
        self.retry = retry
        self.conn = httplib.HTTPConnection(host, port, timeout=None)

    def reconnect(self):
        self.conn.close()
        self.conn.connect()

    def request(self, verb, url, body=None):
        attempt = 0
        while attempt < self.retry:
            attempt += 1
            try:
                self.conn.request(verb, url, body)
                response = self.conn.getresponse()
                response_body = response.read()
                response_status = response.status
                if 200 == response_status:
                    return response_body
                elif 404 == response_status:
                    raise KeyError()
                else:
                    raise HTTPError(response.status, response_body)
            except (socket.error, httplib.ImproperConnectionState, httplib.BadStatusLine):
                self.reconnect()

    def get(self, key):
        return self.request("GET", "/%s" % key)

    def put(self, key, value):
        return self.request("PUT", "/%s" % key, value)


class Client(object):
    def __init__(self):
        raise NotImplementedError()

#endregion

# region Tornado

try:
    import tornado
except ImportError, ex:
    class TornadoAsyncClient:
        def __init__(self):
            raise ex
else:
    import tornado.httpclient

    try:
        from tornado.curl_httpclient import CurlAsyncHTTPClient as AsyncHttpClient
    except ImportError:
        from tornado.httpclient import AsyncHTTPClient as AsyncHttpClient

    class TornadoAsyncClient(object):
        def __init__(self, port=6521, host="127.0.0.1"):
            self.url = "http://%s:%s/%%s" % (host, port)
            self.client = AsyncHttpClient()

        def get(self, key, callback):
            def _callback(response):
                if 200 == response.code:
                    callback(response.body)
                elif 404 == response.code:
                    callback(None)
                raise Exception(response.error)

            self.client.fetch(self.url % key, method="GET", callback=_callback)

        def put(self, key, value):
            self.client.fetch(self.url % key, method="PUT", body=value)

# endregion

__all__ = ["Client", "Connection", "TornadoAsyncClient"]