import urllib
try:
    import ujson as json
except ImportError:
    import json


class HTTPError(Exception):
    def __init__(self, code, message):
        self.code = code
        self.message = message

    def __str__(self):
        return "\nResponse code: %s\nResponse: %s" % (self.code, self.message)


DEFAULT_PORT = 6521
DEFAULT_HOST = "127.0.0.1"

#region Default

import httplib
import socket


class Connection(object):
    def __init__(self, port=DEFAULT_PORT, host=DEFAULT_HOST, reconnect=1):
        self.url = "http://%s:%s/%%s" % (host, port)
        self.reconnect = reconnect
        self.conn = httplib.HTTPConnection(host, port, timeout=None)

    def reconnect(self):
        self.conn.close()
        self.conn.connect()

    def request(self, verb, url, body=None):
        attempt = 0
        while attempt < self.reconnect:
            attempt += 1
            try:
                self.conn.request(verb, url, body)
                response = self.conn.getresponse()
                response_body = response.read()
                response_status = response.status
                if 200 == response_status:
                    return response_body
                elif 404 == response_status:
                    raise KeyError()
                else:
                    raise HTTPError(response.status, response_body)
            except (socket.error, httplib.ImproperConnectionState, httplib.BadStatusLine):
                self.reconnect()

    def get(self, key):
        return self.request("GET", "/%s" % key)

    def put(self, key, value):
        return self.request("PUT", "/%s" % key, value)

    def _get(self, key, callback):
        callback(self.get(key))

    def _put(self, key, value, callback):
        callback(self.put(key, value))


class Client(object):
    def __init__(self):
        raise NotImplementedError()

#endregion

# region Tornado

try:
    import tornado
except ImportError, ex:
    tornado = False

    class TornadoAsyncClient:
        def __init__(self, *args, **kwargs):
            raise ex
else:
    import tornado.httpclient

    try:
        from tornado.curl_httpclient import CurlAsyncHTTPClient as AsyncHttpClient
    except ImportError:
        from tornado.httpclient import AsyncHTTPClient as AsyncHttpClient

    class TornadoAsyncClient(object):
        def __init__(self, port=DEFAULT_PORT, host=DEFAULT_HOST):
            self.root_url = "http://%s:%s" % (host, port)
            self.url = "%s/%%s" % self.root_url
            self.client = AsyncHttpClient()
            self._get = self.get
            self._put = self.put

        def get(self, key, callback):
            def _callback(response):
                if 200 == response.code:
                    callback(response.body)
                elif 404 == response.code:
                    callback(None)
                raise Exception(response.error)

            self.client.fetch(self.url % key, method="GET", callback=_callback)

        def put(self, key, value, callback=None):
            self.client.fetch(self.url % key, method="PUT", body=value, callback=callback)

        def get_multiple(self, keys, callback=None):
            def _callback(response):
                callback(json.loads(response.body))

            self.client.fetch(self.url % ("?%s" % urllib.urlencode({"key": keys}, True)),
                              method="GET", body=json.dumps(keys), callback=_callback)

        def get_multiple_mapped(self, keys, callback):
            def _callback(values):
                mapping = {}
                i = iter(values)
                for key in keys:
                    value = i.next()
                    mapping[key] = value
                callback(mapping)

            self.get_multiple(keys, _callback)

        def put_multiple(self, mapping, callback=None):
            self.client.fetch(self.root_url, method="PUT", body=json.dumps(mapping), callback=callback)

# endregion

#region Redis

try:
    import redis

    class RedisCacheClient():
        def __init__(self, leveldb, redis, ttl=3600):
            self.leveldb = leveldb
            self.redis = redis
            self.ttl = ttl

            self._redis_set = s = redis.register_script("""
                redis.call("SET", KEYS[1], KEYS[2])
                redis.call("EXPIRE", KEYS[1], KEYS[3])
            """)
            redis.set_ttl = lambda key, value, ttl=ttl, client=None: s([key, value, ttl], client=client or redis)

        def get(self, key, ttl=None, *args):
            value = self.redis.get(key)
            if value is None:
                try:
                    value = self.leveldb.get(key)
                    self._redis_set([key, value, ttl or self.ttl])
                except KeyError:
                    pass

            return value

        def set(self, key, value, ttl=None, *args):
            self.leveldb.put(key, value)
            self._redis_set([key, value, ttl or self.ttl])

    class RedisAsyncCacheClient(RedisCacheClient):
        def get(self, key, ttl=None, callback=None):
            def got_value(value):
                self._redis_set([key, value, ttl or self.ttl])
                callback(value)

            value = self.redis.get(key)
            if value is None:
                try:
                    return self.leveldb.get(key, got_value)
                except KeyError:
                    pass

            callback(value)

        def set(self, key, value, ttl=None, callback=None):
            def value_was_set():
                self._redis_set([key, value, ttl or self.ttl])
                callback(True)

            self.leveldb.put(key, value, value_was_set)


except ImportError, ex:
    redis = False

#endregion

__all__ = ["Client", "Connection", "TornadoAsyncClient", "RedisCacheClient", "RedisAsyncCacheClient"]