__author__ = 'ardevelop'

import leveldb
import tornado.web
import tornado.ioloop


class Application(tornado.web.Application):
    def __init__(self, path):
        self.db = leveldb.LevelDB(path)
        self.get = 0
        self.miss = 0
        self.put = 0
        super(Application, self).__init__([("/", StatsHandler), ("/(.+)", RequestHandler)], auto_etag=False)

    def log_request(self, handler):
        pass


class RequestHandler(tornado.web.RequestHandler):
    def get(self, key):
        app = self.application
        try:
            self.finish(app.db.Get(key))
            app.get += 1
        except KeyError:
            app.miss += 1
            raise tornado.web.HTTPError(404)

    def put(self, key):
        value = self.request.body
        if value:
            app = self.application
            app.db.Put(key, value)
            app.put += 1

    def write_error(self, status_code, **kwargs):
        self.finish()


class StatsHandler(tornado.web.RequestHandler):
    def get(self):
        app = self.application
        self.finish("get: %s, put: %s, miss: %s" % (app.get, app.put, app.miss))


def run():
    import sys
    import argparse
    import ardaemon

    parser = argparse.ArgumentParser(description="leveldb server")
    parser.add_argument("-i", metavar="address", type=str, default="0.0.0.0", dest="interface",
                        help="listening interface")
    parser.add_argument("-p", metavar="port", type=int, default=6521, dest="port", help="listening port")
    parser.add_argument("-d", metavar="path", type=str, dest="path", default="/tmp/leveldb", help="database location")

    with ardaemon.Daemon(name="leveldb", parser=parser, stdout=sys.stdout, stderr=sys.stderr) as daemon:
        import tornado.httpserver
        import tornado.ioloop

        app = Application(daemon.args.path)
        srv = tornado.httpserver.HTTPServer(app)
        srv.bind(daemon.args.port, daemon.args.interface)
        srv.start()
        tornado.ioloop.IOLoop.instance().start()
