# -*- coding: utf-8 -*-
import json
from pagador.acesso.externo import FormatoDeEnvio
from pagador.retorno.models import SituacaoPedido
from pagador.retorno.registro import RegistroBase


class SituacoesDePagamento(object):
    aprovado = "approved"
    pendente = "pending"
    em_andamento = "in_process"
    rejeitado = "rejected"
    devolvido = "refunded"
    cancelado = "cancelled"
    em_mediacao = "in_mediation"
    charged_back = "charged_back"

    @classmethod
    def do_tipo(cls, tipo):
        return getattr(cls, tipo, None)


class Topicos(object):
    pagamento = "payment"
    ordem_pagamento = "merchant_order"


class Registro(RegistroBase):
    def __init__(self, dados, tipo="retorno", configuracao=None):
        super(Registro, self).__init__(dados, configuracao)
        self.exige_autenticacao = True
        self.processa_resposta = True
        self.tipo = tipo
        self.formato_de_envio = FormatoDeEnvio.json

    @property
    def url(self):
        if self.topico == Topicos.pagamento:
            return "https://api.mercadolibre.com/collections/notifications/{}".format(self.dados["id"])
        if self.topico == Topicos.ordem_pagamento:
            return self.dados.get("resource", "https://api.mercadolibre.com/merchant_orders/{}".format(self.dados["id"]))

    @property
    def pedido_numero(self):
        if self.retorno_de_notificacao:
            return self.item_topico["external_reference"]
        if "external_reference" in self.dados:
            return self.dados["external_reference"]
        return None

    @property
    def identificador_id(self):
        if self.retorno_de_preferencia:
            return self.dados["preference_id"]
        return None

    def __getattr__(self, name):
        if name.startswith("situacao_"):
            tipo = name.replace("situacao_", "")
            if self.retorno_de_preferencia:
                if tipo != "aprovado" and tipo != "pendente":
                    return False
                if self.dados["collection_status"] is None:
                    return False
                return self.dados["collection_status"] == SituacoesDePagamento.do_tipo(tipo)
            if self.retorno_de_notificacao:
                return self.item_topico["status"] == SituacoesDePagamento.do_tipo(tipo)
        return object.__getattribute__(self, name)

    @property
    def situacao_do_pedido(self):
        if self.situacao_aprovado:
            return SituacaoPedido.SITUACAO_PEDIDO_PAGO
        if self.situacao_em_andamento or self.situacao_pendente:
            return SituacaoPedido.SITUACAO_PAGTO_EM_ANALISE
        if self.situacao_rejeitado:
            return SituacaoPedido.SITUACAO_AGUARDANDO_PAGTO
        if self.situacao_devolvido:
            return SituacaoPedido.SITUACAO_PAGTO_DEVOLVIDO
        if self.situacao_cancelado:
            return SituacaoPedido.SITUACAO_PEDIDO_CANCELADO
        if self.situacao_em_mediacao:
            return SituacaoPedido.SITUACAO_PAGTO_EM_DISPUTA
        if self.situacao_charged_back:
            return SituacaoPedido.SITUACAO_PAGTO_CHARGEBACK
        return SituacaoPedido.SITUACAO_AGUARDANDO_PAGTO

    @property
    def alterar_situacao(self):
        return True

    @property
    def retorno_de_preferencia(self):
        return "collection_status" in self.dados

    @property
    def retorno_de_notificacao(self):
        return self.chave_topico in self.dados

    @property
    def obter_dados_do_gateway(self):
        if not self.dados.get("conta_id", None):
            return False
        return self.topico is not None

    @property
    def deve_gravar_dados_de_pagamento(self):
        return self.obter_dados_do_gateway

    @property
    def valores_de_pagamento(self):
        valores = {
            "transacao_id": self.item_topico["id"]
        }
        if "total_paid_amount" in self.item_topico:
            valores["valor_pago"] = self.item_topico["total_paid_amount"]
        return valores

    @property
    def chave_topico(self):
        if self.topico == Topicos.pagamento:
            return "collection"
        if self.topico == Topicos.ordem_pagamento:
            return "payments"

    @property
    def item_topico(self):
        if self.topico == Topicos.pagamento:
            return self.dados[self.chave_topico]
        if self.topico == Topicos.ordem_pagamento:
            return self.dados[self.chave_topico][0]

    @property
    def topico(self):
        if "topic" in self.dados:
            return self.dados["topic"]
        return None

    @property
    def redireciona_para(self):
        if "next_url" in self.dados:
            return "{}?{}=1".format(self.dados["next_url"], self.tipo)
        return None

    def processar_resposta(self, resposta):
        retorno = json.loads(resposta.content)
        if resposta.status_code == 401:
            reenviar = retorno["message"] == "expired_token"
            return {"content": retorno["message"], "status": 401 if reenviar else 400, "reenviar": reenviar}
        if resposta.status_code == 400:
            reenviar = retorno["error"] == "invalid_access_token"
            return {"content": retorno["message"], "status": 401 if reenviar else 400, "reenviar": reenviar}
        return {"content": retorno, "status": resposta.status_code, "reenviar": False}

