# -*- coding: utf-8 -*-
import json
from urllib import urlencode
from datetime import timedelta, datetime

import requests
from pagador.seguranca import autenticador
from pagador.seguranca.autenticador import TipoAutenticacao
from pagador.seguranca.instalacao import Parametros, InstalacaoNaoFinalizada, InstaladorBase
from pagador.settings import MERCADOPAGO_REDIRECT_URL
import pagador_mercadopago


class ParametrosMercadoPago(Parametros):
    @property
    def chaves(self):
        return ["client_secret", "client_id", "sponsor_id"]


class TipoToken(object):
    authorization_code = "authorization_code"
    refresh_token = "refresh_token"


class Credenciador(autenticador.Credenciador):
    def __init__(self, configuracao):
        self.conta_id = configuracao.conta_id
        self.configuracao = configuracao
        self.access_token = getattr(configuracao, "token", "")
        self.refresh_token = getattr(configuracao, "codigo_autorizacao", "")
        self.token_expiracao = getattr(configuracao, "token_expiracao", datetime.now())
        self.tipo = TipoAutenticacao.query_string

    def _atualiza_credenciais(self):
        self.access_token = getattr(self.configuracao, "token", "")
        self.refresh_token = getattr(self.configuracao, "codigo_autorizacao", "")
        self.token_expiracao = getattr(self.configuracao, "token_expiracao", datetime.now())

    def obter_credenciais(self):
        self._atualiza_credenciais()
        return self.access_token

    def esta_valido(self):
        if not self.access_token:
            return False
        return datetime.now() > self.token_expiracao


class Instalador(InstaladorBase):
    campos = ["usuario", "token", "token_expiracao", "codigo_autorizacao"]

    def __init__(self, configuracao, **filtro_parametros):
        super(Instalador, self).__init__(configuracao)
        self.conta_id = filtro_parametros["id"]
        self.parametros = ParametrosMercadoPago("mercadopago", **filtro_parametros)

    def url_ativador(self, parametros_redirect):
        dados = {
            'response_type': 'code',
            'client_id': self.parametros.client_id,
            'redirect_uri': "{}?{}".format(MERCADOPAGO_REDIRECT_URL.format(self.conta_id), urlencode(parametros_redirect))
        }
        return 'http://auth.mercadolivre.com.br/authorization?{}'.format(urlencode(dados))

    def obter_dados(self, dados, tipo=TipoToken.authorization_code):
        dados.update({
            "grant_type": tipo,
            "client_id": self.parametros.client_id,
            "client_secret": self.parametros.client_secret,
        })
        if tipo == TipoToken.authorization_code:
            dados["redirect_uri"] = "{}?{}".format(MERCADOPAGO_REDIRECT_URL.format(self.conta_id), urlencode({"next_url": dados["next_url"]}))
            del dados["next_url"]

        url = "https://api.mercadolibre.com/oauth/token"
        headers = {
            'Accept': 'application/json',
            'User-Agent': 'LI-Pagador_MercadoPago-{}v'.format(pagador_mercadopago.__version__),
            'Content-type': 'application/json'
        }
        resposta = requests.post(url, params=dados, headers=headers)
        if resposta.status_code == 200:
            return json.loads(resposta.content)
        return {"erro": resposta.content}

    def obter_user_id(self, access_token):
        url = "https://api.mercadolibre.com/users/me?access_token={}".format(access_token)
        resposta = requests.get(url)
        if resposta.status_code == 200:
            resultado = json.loads(resposta.content)
            return resultado["id"]
        return None

    def dados_de_instalacao(self, dados):
        dados_instalacao = self.obter_dados(dados)
        user_id = self.obter_user_id(dados_instalacao["access_token"])
        if "erro" in dados_instalacao:
            raise InstalacaoNaoFinalizada(dados_instalacao["erro"])
        return {
            "usuario": user_id,
            "token": dados_instalacao["access_token"],
            "token_expiracao": datetime.now() + timedelta(seconds=dados_instalacao["expires_in"]),
            "codigo_autorizacao": dados_instalacao["refresh_token"]
        }

    def dados_de_atualizacao(self, dados):
        access_token = self.obter_dados({"refresh_token": dados["codigo_autorizacao"]}, tipo=TipoToken.refresh_token)
        if "erro" in access_token:
            raise InstalacaoNaoFinalizada(access_token["erro"])
        return {
            "token": access_token["access_token"],
            "token_expiracao": datetime.now() + timedelta(seconds=access_token["expires_in"]),
            "codigo_autorizacao": access_token["refresh_token"]
        }

    def desinstalar(self, dados):
        url = "https://api.mercadolibre.com/users/{}/applications/{}?access_token={}".format(dados["usuario"], self.parametros.client_id, dados["token"])
        resposta = requests.delete(url)
        if resposta.status_code == 200:
            return json.loads(resposta.content)
        return {"erro": resposta.content}
