# -*- coding: utf-8 -*-
import urllib
from pagador import settings
from pagador.acesso.externo import FormatoDeEnvio
from pagador.retorno.models import SituacaoPedido, Pedido
from pagador.retorno.registro import RegistroBase
from pagador_paypal.extensao.envio import ExpressCheckout
from pagador_paypal.extensao.seguranca import ParametrosPayPal


class SituacoesDePagamento(object):
    aprovado = "completed"
    pendente = "pending"
    em_andamento = "?"
    rejeitado = "?"
    devolvido = "refunded"
    cancelado = "?"
    em_mediacao = "?"
    charged_back = "?"

    @classmethod
    def do_tipo(cls, tipo):
        return getattr(cls, tipo, None)


class AckRetorno(object):
    def __init__(self, ack_code):
        ack = ack_code.lower()
        self.sucesso = ack in ("success", "successwithwarning")
        self.failure = ack == "failure"


class Registro(RegistroBase):
    def __init__(self, dados, tipo="retorno", configuracao=None):
        super(Registro, self).__init__(dados, configuracao)
        self.exige_autenticacao = True
        self.processa_resposta = True
        self.tipo = tipo
        self._pedido = None
        self.formato_de_envio = FormatoDeEnvio.form_urlencode
        self.sucesso = tipo == "success"
        self._valores_de_pagamento = {}

    @property
    def sandbox(self):
        return "sandbox." if (settings.ENVIRONMENT == "local" or settings.ENVIRONMENT == "development") else ""

    @property
    def url(self):
        return "https://api-3t.{}paypal.com/nvp".format(self.sandbox)

    @property
    def pedido_numero(self):
        if "referencia" in self.dados:
            return self.dados["referencia"]
        if "paymentrequest_0_invnum" in self.dados:
            return self.dados["paymentrequest_0_invnum"]
        if "invoice" in self.dados:
            return self.dados["invoice"]
        return None

    @property
    def identificador_id(self):
        return self.dados.get("paymentinfo_0_transactionid", None)

    @property
    def chave_status(self):
        if "payment_status" in self.dados:
            return "payment_status"
        if "paymentinfo_0_paymentstatus" in self.dados:
            return "paymentinfo_0_paymentstatus"
        return None

    def __getattr__(self, name):
        if name.startswith("situacao_"):
            tipo = name.replace("situacao_", "")
            if self.chave_status:
                return self.dados[self.chave_status].lower() == SituacoesDePagamento.do_tipo(tipo)
            return False
        return object.__getattribute__(self, name)

    @property
    def situacao_do_pedido(self):
        if self.situacao_aprovado:
            return SituacaoPedido.SITUACAO_PEDIDO_PAGO
        if self.situacao_em_andamento or self.situacao_pendente:
            return SituacaoPedido.SITUACAO_PAGTO_EM_ANALISE
        if self.situacao_rejeitado:
            return SituacaoPedido.SITUACAO_AGUARDANDO_PAGTO
        if self.situacao_devolvido:
            return SituacaoPedido.SITUACAO_PAGTO_DEVOLVIDO
        if self.situacao_cancelado:
            return SituacaoPedido.SITUACAO_PEDIDO_CANCELADO
        if self.situacao_em_mediacao:
            return SituacaoPedido.SITUACAO_PAGTO_EM_DISPUTA
        if self.situacao_charged_back:
            return SituacaoPedido.SITUACAO_PAGTO_CHARGEBACK
        return SituacaoPedido.SITUACAO_AGUARDANDO_PAGTO

    @property
    def alterar_situacao(self):
        return self.chave_status in self.dados

    @property
    def grava_identificador(self):
        return "paymentinfo_0_transactionid" in self.dados

    @property
    def valores_de_pagamento(self):
        return self._valores_de_pagamento

    @property
    def retorno_de_requisicao(self):
        return self.tipo != "retorno"

    @property
    def retorno_de_notificacao(self):
        return self.tipo == "retorno"

    @property
    def obter_dados_do_gateway(self):
        if self.retorno_de_requisicao:
            return self.tipo in ["success", "pending"]
        return False

    @property
    def redireciona_para(self):
        if "next_url" in self.dados:
            if self.sucesso:
                return "{}?{}=1".format(self.dados["next_url"], self.tipo)
            else:
                return "{}?failure=1&erro_retorno=1".format(self.dados["next_url"])
        return None

    def processar_resposta(self, resposta):
        content = resposta.content
        retorno = {}
        if content:
            content = urllib.unquote(content).decode('utf8')
            retorno = {par.split("=")[0].lower(): par.split("=")[1].lower() for par in content.split("&")}
        ack = AckRetorno(retorno["ack"])
        self.sucesso = ack.sucesso
        if "paymentinfo_0_paymentstatus" in retorno and retorno["paymentinfo_0_paymentstatus"].lower() == SituacoesDePagamento.do_tipo("pendente"):
            self.tipo = "pending"
        if "paymentinfo_0_transactionid" in retorno:
            self._valores_de_pagamento["transacao_id"] = retorno["paymentinfo_0_transactionid"].upper()
            self._valores_de_pagamento["identificador_id"] = retorno["token"].upper()
        if "paymentinfo_0_amt" in retorno:
            self._valores_de_pagamento["valor_pago"] = self.formatador.string_para_decimal(retorno["paymentinfo_0_amt"])
        return {"content": retorno, "status": 200 if self.sucesso else 500, "reenviar": False}

    @property
    def pedido(self):
        if not self._pedido:
            if not "conta_id" in self.dados:
                return None
            conta_id = int(self.dados["conta_id"])
            try:
                self._pedido = Pedido.objects.get(conta_id=conta_id, numero=self.pedido_numero)
            except Pedido.DoesNotExist:
                return None
        return self._pedido

    def gerar_dados_de_envio(self):
        notification_url = settings.PAYPAL_NOTIFICATION_URL.format(self.pedido.conta_id)
        parametros = ParametrosPayPal("paypal", id=self.pedido.conta_id)
        ExpressCheckout.cria_payment_request()
        ExpressCheckout.cria_shipto()
        for item in range(0, len(self.pedido.itens.all())):
            ExpressCheckout.cria_item_payment_request(item)
        request_payment = ExpressCheckout(
            user=parametros.username,
            pwd=parametros.password,
            signature=parametros.signature,
            notifyurl=notification_url,
            version=settings.PAYPAL_VERSION,
            method='DoExpressCheckoutPayment',
            token=self.dados["token"],
            payerid=self.dados["PayerID"],

            paymentrequest_0_paymentaction='SALE',
            paymentrequest_0_notifyurl=notification_url,
            paymentrequest_0_amt=self.formatador.formata_decimal(self.pedido.valor_total),
            paymentrequest_0_shippingamt=self.formatador.formata_decimal(self.pedido.valor_envio),
            paymentrequest_0_currencycode="BRL",
            paymentrequest_0_itemamt=self.formatador.formata_decimal(self.pedido.valor_subtotal),
            paymentrequest_0_invnum=self.pedido.numero,

            paymentrequest_0_shiptoname=self.pedido.endereco_entrega.nome,
            paymentrequest_0_shiptostreet="{}, {} {}".format(self.pedido.endereco_entrega.endereco, self.pedido.endereco_entrega.numero, self.pedido.endereco_entrega.complemento),
            paymentrequest_0_shiptostreet2=self.pedido.endereco_entrega.bairro,
            paymentrequest_0_shiptocity=self.pedido.endereco_entrega.cidade,
            paymentrequest_0_shiptostate=self.pedido.endereco_entrega.estado,
            paymentrequest_0_shiptozip=self.pedido.endereco_entrega.cep,
            paymentrequest_0_shiptocountrycode="BR",
            paymentrequest_0_shiptophonenum=self.pedido.cliente.telefone_principal
        )
        for indice, item in enumerate(self.pedido.itens.all()):
            self.define_valor_de_atributo_de_item(request_payment, "NAME", indice, item.nome[:127])
            self.define_valor_de_atributo_de_item(request_payment, "DESC", indice, item.produto.descricao_completa[:127])
            self.define_valor_de_atributo_de_item(request_payment, "AMT", indice, self.formatador.formata_decimal(item.preco_venda))
            self.define_valor_de_atributo_de_item(request_payment, "QTY", indice, self.formatador.formata_decimal(item.quantidade))
            self.define_valor_de_atributo_de_item(request_payment, "NUMBER", indice, item.sku[:127])
            self.define_valor_de_atributo_de_item(request_payment, "ITEMURL", indice, item.produto.get_absolute_url())

        return request_payment.to_dict()

    def define_valor_de_atributo_de_item(self, request, atributo, indice, valor):
        nome = "L_PAYMENTREQUEST_0_{}{}".format(atributo, indice)
        request.define_valor_de_atributo(nome, {nome.lower(): valor})
