# -*- coding: utf-8 -*-
from pagador.configuracao.cadastro import ValoresBase

from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from repositories.configuracao import models


class Gateways(object):
    pagseguro = 1
    pagamento_digital = 2
    paypal = 3
    mercado_pago = 4
    koin = 9


class FormaDePagamento(models.FormaPagamento, Plugavel):
    class Meta:
        proxy = True

    def save(self, *args, **kwargs):
        if self.conta and not self.contrato_id:
            self.contrato_id = self.conta.contrato_id
        super(FormaDePagamento, self).save(*args, **kwargs)

    @property
    def imagem_admin(self):
        return 'img/formas-de-pagamento/{}-logo.png'.format(self.codigo)

    @property
    def imagem_loja(self):
        return 'img/bandeiras/{}-logo.png'.format(self.codigo)

    @property
    def html_escolha(self):
        try:
            return self.plugin_configuracao.MeioPagamentoSelecao()
        except AttributeError:
            return ""

    @property
    def em_uso_para_o_usuario(self):
        try:
            return (self.pagamento_configuracao_usuario or
                    self.pagamento_configuracao_senha or
                    self.pagamento_configuracao_assinatura or
                    self.pagamento_configuracao_token or
                    self.pagamento_configuracao_codigo_autorizacao or
                    self.bancos.all or self.pagamento_configuracao_json)
        except AttributeError:
            return None

    @classmethod
    def configuracoes_ativas(cls, conta_id=None):
        return cls.objects.raw(
            """SELECT
                    pg.pagamento_id,
                    pg.pagamento_nome,
                    pg.pagamento_codigo,
                    pg.pagamento_ativado,
                    pgc.pagamento_configuracao_id,
                    pgc.pagamento_configuracao_usuario,
                    pgc.pagamento_configuracao_senha,
                    pgc.pagamento_configuracao_token,
                    pgc.pagamento_configuracao_assinatura,
                    pgc.pagamento_configuracao_codigo_autorizacao,
                    pgc.pagamento_configuracao_ativo,
                    pgc.pagamento_configuracao_json,
                    pgc.conta_id
               FROM configuracao.tb_pagamento as pg
               LEFT OUTER JOIN configuracao.tb_pagamento_configuracao as pgc on (pgc.pagamento_id = pg.pagamento_id AND pgc.conta_id=%s)
               WHERE pg.pagamento_ativado = true;""", [conta_id]
        )

    def to_dict(self, com_configuracao=True):
        forma_pagamento = {
            "id": self.id,
            "codigo": self.codigo,
            "nome": self.nome,
            "ativo_no_sistema": self.ativo,
            "imagem_admin": self.imagem_admin,
            "imagem_loja": self.imagem_loja,
            "html_escolha": self.html_escolha,
            "plano_indice": self.plano_indice,
        }
        if com_configuracao:
            forma_pagamento["ativo_para_a_conta"] = self.pagamento_configuracao_ativo
            forma_pagamento["configurado"] = self.em_uso_para_o_usuario is not None

        return forma_pagamento


class FormaDePagamentoConfiguracao(models.FormaPagamentoConfiguracao, Plugavel):
    class Meta:
        proxy = True

    @property
    def meio_pagamento(self):
        return self.forma_pagamento.codigo

    def save(self, *args, **kwargs):
        self.desconto_tipo = self.TIPO_PORCENTAGEM
        self.desconto = self.desconto_valor and True or False
        if self.maximo_parcelas != 0 and self.parcelas_sem_juros > self.maximo_parcelas:
            self.parcelas_sem_juros = self.maximo_parcelas

        if self.conta and not self.contrato_id:
            self.contrato_id = self.conta.contrato_id

        if self.email_comprovante and self.conta_id and self.conta.email == self.email_comprovante:
            self.email_comprovante = None

        super(FormaDePagamentoConfiguracao, self).save(*args, **kwargs)

    @property
    def formulario(self):
        if not hasattr(self, "_formulario"):
            self._formulario = None
        if not self._formulario:
            try:
                self._formulario = self.plugin_configuracao.Formulario()
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento {} (id={}) não possui a classe configuracao.Formulario.".format(self.forma_pagamento.codigo, self.forma_pagamento.id))
        return self._formulario

    @property
    def script(self):
        try:
            return self.plugin_configuracao.MeioPagamentoEnvio()
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoEnvio.".format(self.forma_pagamento.codigo))

    @property
    def selecao(self):
        try:
            return self.plugin_configuracao.MeioPagamentoSelecao()
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoSelecao.".format(self.forma_pagamento.codigo))

    @property
    def ativador(self):
        try:
            return self.plugin_seguranca.Ativador()
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe seguranca.Ativador.".format(self.forma_pagamento.codigo))

    @property
    def valores(self):
        try:
            return ValoresBase(self, self.formulario)
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoValores.".format(self.forma_pagamento.codigo))

    @property
    def cadastro(self):
        try:
            return self.plugin_configuracao.MeioPagamentoCadastro()
        except AttributeError:
            return None
            #FIXME: Pensar em um modelo melhor para determinar extensoes não obrigatórias
            # raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoCadastro.".format(self.forma_pagamento.codigo))

    def salvar(self, dados):
        self.formulario.define_valores_em_model(self, dados).save()

    def to_dict(self):
        cadastro = None
        if self.cadastro:
            cadastro = self.cadastro.to_dict()
        return {
            "cadastro": cadastro,
            "valores": self.valores.to_dict(),
            "formulario": self.formulario.to_dict()
        }
