# -*- coding: utf-8 -*-
from pagador.pedido import notificador
from pagador.envio.models import Pedido
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from pagador.retorno.models import SituacaoPedido
from pagador.seguranca.autenticador import Conexao, TipoConexao


class Enviar(object):
    @property
    def utils(self):
        class Utils(object):
            @classmethod
            def formata_data(cls, data, hora=True, iso=False):
                if hora:
                    return data.strftime("%Y-%m-%d{}%H:%M:%S".format("T" if iso else " "))
                return data.strftime("%Y-%m-%d")

            @classmethod
            def formata_decimal(cls, valor, como_float=False):
                if como_float:
                    return float('{0:.3g}'.format(valor))
                return '{0:.3g}'.format(valor)

            @classmethod
            def converte_tel_em_tupla_com_ddd(cls, telefone):
                return telefone[:2], telefone[2:]
        return Utils

    def obter_situacao_do_pedido(self, status_requisicao):
        if status_requisicao == 200:
            return SituacaoPedido.SITUACAO_PEDIDO_PAGO
        return SituacaoPedido.SITUACAO_PEDIDO_CANCELADO


class EnviarPedido(Plugavel):
    def __init__(self, meio_pagamento, pedido_numero, dados, conta_id, configuracao_pagamento=None):
        self.meio_pagamento = meio_pagamento
        self.pedido = Pedido.objects.get(numero=pedido_numero, conta_id=conta_id)
        self.dados = dados
        self.configuracao_pagamento = configuracao_pagamento
        self.headers = {"Content-Type": "application/json"}
        self._requisicao = None

    def processar_resposta(self, resposta):
        if getattr(self.requisicao, "processa_resposta", False):
            return self.requisicao.processar_resposta(resposta)
        return {"data": resposta.content, "status": resposta.status_code, "reenviar": False}

    @property
    def requisicao(self):
        if not self._requisicao:
            try:
                self._requisicao = self.plugin_requisicao.EnviarPedido(self.pedido, self.dados)
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe requisicao.EnviarPedido.".format(self.meio_pagamento))
        return self._requisicao

    def post(self, segunda_tentativa=False):
        conexao = Conexao(self.meio_pagamento, self.configuracao_pagamento, self.requisicao, TipoConexao.envio, self.processar_resposta)
        return conexao.post(segunda_tentativa)

    def enviar(self):
        resultado = self.post()
        if resultado.get("reenviar", False):
            resultado = self.post(segunda_tentativa=True)
        if resultado["status"] == 201 or resultado["status"] == 200:
            if self.requisicao.grava_identificador:
                self.grava_identificacao_de_pagamento(resultado["identificador"])
            notificador.envia_alteracao_de_situacao_de_pedido(self.pedido.numero, self.requisicao.obter_situacao_do_pedido(resultado["status"]), self.pedido.conta.id)
        return resultado

    def grava_identificacao_de_pagamento(self, identificacao):
        pedido_pagamento = self.pedido.pedido_pagamentos.get(pagamento=self.pedido.pagamento)
        pedido_pagamento.identificador_id = identificacao
        pedido_pagamento.save()