# -*- coding: utf-8 -*-

from pagador import settings
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from pagador.pedido import notificador
from pagador.retorno.models import PedidoPagamento
from pagador.retorno.notificador import grava_evidencia_retorno
from pagador.seguranca.autenticador import Conexao, TipoConexao


class RegistroBase(object):
    def __init__(self, dados):
        self.dados = dados
        self.exige_autenticacao = False
        self.processa_resposta = False

    @property
    def url(self):
        raise NotImplementedError

    @property
    def pedido_numero(self):
        raise NotImplementedError

    @property
    def identificador_id(self):
        raise NotImplementedError

    @property
    def situacao_do_pedido(self):
        raise NotImplementedError

    @property
    def alterar_situacao(self):
        raise NotImplementedError

    @property
    def redireciona_para(self):
        raise NotImplementedError

    def processar_resposta(self, resposta):
        raise NotImplementedError


class RegistrarPagamento(Plugavel):
    def __init__(self, meio_pagamento, dados, tipo="retorno", configuracao_pagamento=None):
        self.meio_pagamento = meio_pagamento
        self.dados = dados
        self._registrador = None
        self._conta_id = None
        self.tipo = tipo
        self.configuracao_pagamento = configuracao_pagamento

    @property
    def registrador(self):
        if not self._registrador:
            try:
                self._registrador = self.plugin_retorno.Registro(self.dados, self.tipo)
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe retorno.Registro.".format(self.meio_pagamento))
        return self._registrador

    @property
    def conta_id(self):
        if self._conta_id:
            return self._conta_id
        if self.configuracao_pagamento:
            self._conta_id = self.configuracao_pagamento.conta.id
        if self.registrador.identificador_id:
            try:
                pedido_pagamento = PedidoPagamento.objects.get(identificador_id=self.registrador.identificador_id)
                self._conta_id = pedido_pagamento.conta.id
            except PedidoPagamento.DoesNotExist, PedidoPagamento.MultipleObjectsReturned:
                self._conta_id = None
        return self._conta_id

    def processar_resposta(self, resposta):
        if getattr(self.registrador, "processa_resposta", False):
            return self.registrador.processar_resposta(resposta)
        return {"data": resposta.content, "status": resposta.status_code, "reenviar": False}

    def processar(self):
        resultado = {}
        if settings.GRAVA_EVIDENCIA:
            self.dados["tipo"] = self.tipo
            grava_evidencia_retorno(self.meio_pagamento, self.dados)
            resultado["evidencia"] = True
        if self.registrador.obter_dados_do_gateway:
            self.registrador.dados.update(self.obtem_dados_do_gateway())
        if self.registrador.alterar_situacao:
            if not self.conta_id:
                return {"resultado": "erro", "mensagem": u"Não foi possível alterar o status do pedido, pois a conta não foi identificada com as informações passadas."}
            resultado["situacao"] = u"Alterado para {}".format(self.registrador.situacao_do_pedido)
            notificador.envia_alteracao_de_situacao_de_pedido(self.registrador.pedido_numero, self.registrador.situacao_do_pedido, self.conta_id)
        if self.registrador.redireciona_para:
            return {"redirect": self.registrador.redireciona_para}
        resultado["resultado"] = "OK"
        return resultado

    def get(self, segunda_tentativa=False):
        conexao = Conexao(self.meio_pagamento, self.configuracao_pagamento, self.registrador, TipoConexao.retorno, self.processar_resposta)
        return conexao.get(segunda_tentativa)

    def obtem_dados_do_gateway(self):
        resultado = self.get()
        if resultado.get("reenviar", False):
            resultado = self.get(segunda_tentativa=True)
        if resultado["status"] == 200:
            return resultado["content"]
        return {}