# -*- coding: utf-8 -*-
import json
import traceback
from django.db import IntegrityError
from flask import request, logging, redirect

from flask.ext import restful

from pagador import MeioDePagamentoNaoEncontrado, MeioDePagamentoNaoDefinido
from pagador.configuracao.models import FormaDePagamento, FormaDePagamentoConfiguracao
from pagador.envio.requisicao import EnviarPedido
from pagador.extensao.base import PluginNaoEncontrado
from pagador.retorno.registro import RegistrarPagamento
from api_pagador.app import autenticacao

logger = logging.getLogger("API-PAGADOR")


class HelloWorld(restful.Resource):
    def get(self):
        return {"hello": "world pagador"}


class FormasPagamentoResource(restful.Resource):
    @autenticacao.requerido
    def get(self):
        conta_id = request.args.get('conta_id', None)
        indice = request.args.get('indice', 1)
        com_configuracao = False
        if conta_id:
            com_configuracao = True
            formas_pagamento = FormaDePagamento.configuracoes_ativas(conta_id=conta_id)
        else:
            formas_pagamento = FormaDePagamento.objects.filter(ativo=True)
        return {"formas_pagamento": [forma.to_dict(com_configuracao=com_configuracao, indice=indice) for forma in formas_pagamento]}


class PagamentoBaseResource(restful.Resource):
    def __init__(self):
        self.pagamento_id = None
        self.conta_id = None
        self._forma_de_pagamento = None

    @property
    def dados(self):
        dados = {}
        for arg in request.args:
            dados[arg] = request.args[arg]
        return dados

    def retorno(self, pagamento_id=None):
        if pagamento_id:
            self.pagamento_id = pagamento_id
        if self.retorno_tem_erro:
            status = self.forma_de_pagamento_to_dict["status"]
            del self.forma_de_pagamento_to_dict["status"]
            return self.forma_de_pagamento_to_dict, status
        return self.forma_de_pagamento_to_dict, 200

    @property
    def retorno_tem_erro(self):
        return "erro" in self.forma_de_pagamento_to_dict

    @property
    def forma_de_pagamento(self):
        if not self._forma_de_pagamento:
            try:
                self._forma_de_pagamento = FormaDePagamento.objects.get(id=self.pagamento_id, ativo=True)
            except FormaDePagamento.DoesNotExist:
                self._forma_de_pagamento = None
        return self._forma_de_pagamento

    @property
    def forma_de_pagamento_to_dict(self):
        if not self._forma_de_pagamento:
            try:
                self._forma_de_pagamento = FormaDePagamento.objects.get(id=self.pagamento_id, ativo=True)
                return self._forma_de_pagamento.to_dict(com_configuracao=False)
            except FormaDePagamento.DoesNotExist:
                return {"erro": u"Não foi encontrada forma de pagamento ativa com id {}".format(self.pagamento_id), "status": 404}
        return self._forma_de_pagamento.to_dict(com_configuracao=False)


class FormaPagamentoResource(PagamentoBaseResource):
    @autenticacao.requerido
    def get(self, pagamento_id):
        return self.retorno(pagamento_id)


class PagamentoConfiguracaoBaseResource(PagamentoBaseResource):

    @property
    def chave(self):
        raise NotImplementedError

    @property
    def valor(self):
        raise NotImplementedError

    @property
    def pagamento_configuracao(self):
        if self.conta_id:
            return FormaDePagamentoConfiguracao.objects.get(
                conta_id=self.conta_id,
                forma_pagamento_id=self.pagamento_id
            )
        return None

    @autenticacao.requerido
    def get(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        if self.retorno_tem_erro:
            return self.retorno()
        try:
            return {self.chave: self.valor}
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            logger.error(u"Erro de plugin em {}: {}".format(self.chave, ex))
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            logger.error(u"Erro de banco em {}: {}".format(self.chave, ex))
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500


class FormaPagamentoConfiguracaoResource(PagamentoBaseResource):
    @autenticacao.requerido
    def get(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        if self.retorno_tem_erro:
            return self.retorno()
        try:
            pagamento_configuracao, criado = FormaDePagamentoConfiguracao.objects.get_or_create(
                conta_id=conta_id,
                forma_pagamento_id=pagamento_id
            )
            return pagamento_configuracao.to_dict()
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            logger.error(u"Erro de plugin em {}: {}".format("pagamento configuracao", ex))
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            logger.error(u"Erro de banco em {}: {}".format("pagamento configuracao", ex))
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500

    #FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
    @autenticacao.requerido
    def post(self, pagamento_id, conta_id):
        dados = request.get_json()
        try:
            pagamento_configuracao, criado = FormaDePagamentoConfiguracao.objects.get_or_create(
                conta_id=conta_id,
                forma_pagamento_id=pagamento_id,
            )
            pagamento_configuracao.salvar(dados)
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoScriptResource(PagamentoConfiguracaoBaseResource):

    @property
    def chave(self):
        return "scripts"

    @property
    def valor(self):
        return self.pagamento_configuracao.script.to_dict()


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoSelecaoResource(PagamentoConfiguracaoBaseResource):
    @property
    def chave(self):
        return "scripts"

    @property
    def valor(self):
        return self.pagamento_configuracao.selecao.to_dict()


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoIntegracaoResource(PagamentoConfiguracaoBaseResource):
    @property
    def chave(self):
        return "aplicacao"

    @property
    def valor(self):
        return {"url": self.pagamento_configuracao.instalador.url_ativador(self.dados)}

    @autenticacao.requerido
    def delete(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        try:
            self.pagamento_configuracao.desinstalar()
            return {"resultado": "OK"}, 200
        except Exception, ex:
            return {"erro": u"Ocorreu um erro durante a remoção do aplicativo. Por favor, tente novamente."}, 500
            #PARA DEBUG
            # stack = traceback.format_exc()
            # return {"erro": u"Servidor: {}\n\n{}".format(ex, stack)}, 500


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoInstalacaoResource(PagamentoConfiguracaoBaseResource):
    def get(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        mensagem = None
        next_url = self.dados["next_url"]
        try:
            self.pagamento_configuracao.instalar(self.dados)
        except Exception, ex:
            mensagem = u"Erro: {}".format(ex)
        if mensagem:
            if '?' in next_url:
                return redirect(u"{}&mensagem={}".format(next_url, mensagem), code=301)
            return redirect(u"{}?mensagem={}".format(next_url, mensagem), code=301)
        return redirect(next_url, code=301)


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoConfiguracaoEnviarResource(PagamentoBaseResource):
    @autenticacao.requerido
    def post(self, pagamento_id, conta_id, pedido_numero):
        dados = {}
        if request.data:
            dados = json.loads(request.data)
        self.pagamento_id = pagamento_id
        pagamento = self.forma_de_pagamento_to_dict
        if self.retorno_tem_erro:
            return self.retorno()
        try:
            pagamento_configuracao = FormaDePagamentoConfiguracao.objects.get(
                conta_id=conta_id,
                forma_pagamento_id=pagamento["id"]
            )
            enviar_pedido = EnviarPedido(pagamento["codigo"], pedido_numero, dados, conta_id, pagamento_configuracao)
            return enviar_pedido.enviar()
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500
        except Exception, ex:
            import traceback
            return {"erro": u"Servidor: {} - {}".format(ex, traceback.format_exc())}, 500


class FormaPagamentoRetornoResource(PagamentoConfiguracaoBaseResource):
    def post(self, pagamento_id, conta_id=None, tipo="retorno"):
        return self.get(pagamento_id, conta_id, tipo)

    def get(self, pagamento_id, conta_id=None, tipo="retorno"):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        if self.retorno_tem_erro:
            return self.retorno()
        try:
            registro = RegistrarPagamento(self.forma_de_pagamento.codigo, self.dados, tipo, self.pagamento_configuracao)
            resultado = registro.processar()
            if "redirect" in resultado:
                return redirect(resultado["redirect"], code=301)
            return resultado
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500
        except Exception, ex:
            # import traceback
            return {"erro": u"Servidor: {}".format(ex)}, 500
