# -*- coding: utf-8 -*-

from pagador.configuracao.cadastro import ValoresBase
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from repositories.configuracao import models


class FormaDePagamento(models.FormaPagamento, Plugavel):
    QUERY_CONFIGURACOES_ATIVAS = """SELECT
                pg.pagamento_id,
                pg.pagamento_nome,
                pg.pagamento_codigo,
                pg.pagamento_ativado,
                pgc.pagamento_configuracao_id,
                pgc.pagamento_configuracao_usuario,
                pgc.pagamento_configuracao_senha,
                pgc.pagamento_configuracao_token,
                pgc.pagamento_configuracao_assinatura,
                pgc.pagamento_configuracao_codigo_autorizacao,
                pgc.pagamento_configuracao_ativo,
                pgc.pagamento_configuracao_json,
                pgc.conta_id
           FROM configuracao.tb_pagamento as pg
           LEFT OUTER JOIN configuracao.tb_pagamento_configuracao as pgc on (pgc.pagamento_id = pg.pagamento_id AND pgc.conta_id=%s)
           WHERE pg.pagamento_ativado = true{};"""

    class Meta:
        proxy = True

    @property
    def meio_pagamento(self):
        return self.codigo

    def save(self, *args, **kwargs):
        if self.conta and not self.contrato_id:
            self.contrato_id = self.conta.contrato_id
        super(FormaDePagamento, self).save(*args, **kwargs)

    @property
    def imagem_admin(self):
        return 'img/formas-de-pagamento/{}-logo.png'.format(self.codigo)

    @property
    def imagem_loja(self):
        return 'img/bandeiras/{}-logo.png'.format(self.codigo)

    @property
    def bancos_ativos_na_conta(self):
        bancos_ativos = self.bancos.filter(conta=self.conta, ativo=True)
        if bancos_ativos:
            return [{"id": banco.id, "nome": unicode(banco)} for banco in bancos_ativos]
        return None

    @property
    def configurado(self):
        try:
            try:
                bancos = self.bancos.all()[0]
            except IndexError:
                bancos = None
            return (self.pagamento_configuracao_usuario or
                    self.pagamento_configuracao_senha or
                    self.pagamento_configuracao_assinatura or
                    self.pagamento_configuracao_token or
                    self.pagamento_configuracao_codigo_autorizacao or
                    bancos or self.pagamento_configuracao_json)
        except AttributeError:
            return None

    @classmethod
    def configuracoes_ativas(cls, conta_id, pagamento_id=None):
        parametros = [conta_id]
        if pagamento_id:
            parametros.append(pagamento_id)
        return cls.objects.raw(
            cls.QUERY_CONFIGURACOES_ATIVAS.format((" AND pg.pagamento_id=%s" if pagamento_id else '')), parametros
        )

    def to_dict(self, com_configuracao=True, indice=None):
        forma_pagamento = {
            "id": self.id,
            "codigo": self.codigo,
            "nome": self.nome,
            "ativo_no_sistema": self.ativo,
            "imagem_admin": self.imagem_admin,
            "imagem_loja": self.imagem_loja,
            "plano_indice": self.plano_indice,
            "parcelas": [{"fator": float(parcela.fator), "numero_parcelas": parcela.numero_parcelas} for parcela in self.parcelas.all()],
            "valor_minimo_parcela": float(self.valor_minimo_parcela or 0.0)
        }
        if com_configuracao:
            forma_pagamento["ativo_para_a_conta"] = self.pagamento_configuracao_ativo
            forma_pagamento["configurado"] = self.configurado is not None
            if indice:
                forma_pagamento["habilitado_na_loja"] = forma_pagamento["configurado"] and self.ativo and self.pagamento_configuracao_ativo and self.plano_indice <= indice
                forma_pagamento["bancos_ativos_na_conta"] = self.bancos_ativos_na_conta

        return forma_pagamento


class FormaDePagamentoConfiguracao(models.FormaPagamentoConfiguracao, Plugavel):
    class Meta:
        proxy = True

    @property
    def meio_pagamento(self):
        return self.forma_pagamento.codigo

    def save(self, *args, **kwargs):
        self.desconto_tipo = self.TIPO_PORCENTAGEM
        self.desconto = self.desconto_valor and True or False
        if self.maximo_parcelas and self.parcelas_sem_juros:
            self.parcelas_sem_juros = int(self.parcelas_sem_juros)
            self.maximo_parcelas = int(self.maximo_parcelas)
            if self.maximo_parcelas != 0 and self.parcelas_sem_juros > self.maximo_parcelas:
                self.parcelas_sem_juros = self.maximo_parcelas
        if self.conta and not self.contrato_id:
            self.contrato_id = self.conta.contrato_id

        if self.email_comprovante and self.conta_id and self.conta.email == self.email_comprovante:
            self.email_comprovante = None

        super(FormaDePagamentoConfiguracao, self).save(*args, **kwargs)

    @property
    def formulario(self):
        if not hasattr(self, "_formulario"):
            self._formulario = None
        if not self._formulario:
            try:
                self._formulario = self.plugin_configuracao.Formulario()
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento {} (id={}) não possui a classe configuracao.Formulario.".format(self.forma_pagamento.codigo, self.forma_pagamento.id))
        return self._formulario

    @property
    def script(self):
        try:
            return self.plugin_configuracao.MeioPagamentoEnvio()
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoEnvio.".format(self.forma_pagamento.codigo))

    @property
    def selecao(self):
        try:
            return self.plugin_configuracao.MeioPagamentoSelecao()
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoSelecao.".format(self.forma_pagamento.codigo))

    def instalar(self, dados):
        dados_instalacao = self.instalador.dados_de_instalacao(dados)
        for campo in dados_instalacao.keys():
            setattr(self, campo, dados_instalacao[campo])
        self.save()

    def dados_para_instalador(self):
        dados = {}
        for campo in self.instalador.campos:
            dados[campo] = getattr(self, campo, None)
        return dados

    def atualizar(self):
        dados = self.dados_para_instalador()
        dados_atualizacao = self.instalador.dados_de_atualizacao(dados)
        for campo in dados_atualizacao.keys():
            setattr(self, campo, dados_atualizacao[campo])
        self.save()

    def desinstalar(self):
        dados = self.dados_para_instalador()
        for campo in dados:
            setattr(self, campo, None)
        self.instalador.desinstalar(dados)
        self.save()

    @property
    def instalador(self):
        try:
            return self.plugin_seguranca.Instalador(id=self.conta_id)
        except AttributeError, ex:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe seguranca.Instalador.".format(self.forma_pagamento.codigo))

    @property
    def valores(self):
        try:
            return ValoresBase(self, self.formulario)
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoValores.".format(self.forma_pagamento.codigo))

    @property
    def cadastro(self):
        try:
            return self.plugin_configuracao.MeioPagamentoCadastro(self)
        except AttributeError:
            return None
            #FIXME: Pensar em um modelo melhor para determinar extensoes não obrigatórias
            # raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoCadastro.".format(self.forma_pagamento.codigo))

    @property
    def eh_aplicacao(self):
        try:
            return self.plugin_configuracao.eh_aplicacao
        except AttributeError:
            return False
            #FIXME: Pensar em um modelo melhor para determinar extensoes não obrigatórias

    def salvar(self, dados):
        self.formulario.define_valores_em_model(self, dados).save()

    def to_dict(self):
        cadastro = None
        if self.cadastro:
            cadastro = self.cadastro.to_dict()
        try:
            forma_pagamento = FormaDePagamento.configuracoes_ativas(self.conta.id, pagamento_id=self.forma_pagamento.id)[0].to_dict()
        except IndexError:
            forma_pagamento = {}
        return {
            "cadastro": cadastro,
            "valores": self.valores.to_dict(),
            "formulario": self.formulario.to_dict(),
            "pagamento": forma_pagamento,
            "eh_aplicacao": self.eh_aplicacao
        }
