# -*- coding: utf-8 -*-
import json
import requests
from pagador import settings
from pagador.envio.notificador import grava_evidencia_envio

from pagador.extensao import base


class CredenciadorInvalido(Exception):
    pass


class AutenticadorNaoEncontrado(Exception):
    pass


class Credenciador(object):
    def __init__(self, credenciamento):
        self.credenciamento = credenciamento

    def obter_credenciais(self):
        return "user {token} password {senha}".format(self.credenciamento)

    def esta_valido(self):
        return True

    @property
    def chave(self):
        return "access_token"


class TipoAutenticacao(object):
    cabecalho_http = 1
    query_string = 2


class TipoConexao(object):
    envio = 1
    retorno = 2


class TipoMetodo(object):
    get = "get"
    post = "post"


class Credenciais(base.Plugavel):
    def __init__(self, meio_pagamento, configuracao_pagamento):
        self.meio_pagamento = meio_pagamento
        self.configuracao_pagamento = configuracao_pagamento
        self._credenciador = None

    @property
    def credenciador(self):
        if not self._credenciador:
            try:
                self._credenciador = self.plugin_seguranca.Credenciador(self.configuracao_pagamento)
            except AttributeError:
                raise AutenticadorNaoEncontrado(u"O meio de pagamento '{}' não possui a classe Credenciador.".format(self.meio_pagamento))
        return self._credenciador

    def define_autenticacao(self, url=None, atualiza_autenticacao=False):
        if not url:
            url = self.credenciador.url
        if atualiza_autenticacao or not self.esta_valido():
            self.configuracao_pagamento.atualizar()
        if self.credenciador.tipo == TipoAutenticacao.cabecalho_http:
            return self.por_cabecalho_http()
        if self.credenciador.tipo == TipoAutenticacao.query_string:
            separador = "&" if "?" in url else "?"
            return "{}{}{}".format(url, separador, self.por_query_string())
        return None

    def esta_valido(self):
        return self.credenciador.esta_valido()

    def por_query_string(self):
        try:
            return "{}={}".format(self.credenciador.chave, self.credenciador.obter_credenciais())
        except AttributeError:
            raise CredenciadorInvalido(u"O credenciador {}.{} não implementa o método obter_credenciais(credenciamento=None).".format(self.credenciador.__module__, self.credenciador.__class__.__name__))

    def por_cabecalho_http(self, cabecalho="Authorization"):
        try:
            return {cabecalho: self.credenciador.obter_credenciais()}
        except AttributeError:
            raise CredenciadorInvalido(u"O credenciador {}.{} não implementa o método obter_credenciais(credenciamento=None).".format(self.credenciador.__module__, self.credenciador.__class__.__name__))


class Conexao(object):
    def __init__(self, meio_pagamento, configuracao_pagamento, servico, tipo="envio", processar_resposta=None):
        self.tipo = tipo
        self.meio_pagamento = meio_pagamento
        self.configuracao_pagamento = configuracao_pagamento
        self.servico = servico
        self.headers = {"Content-Type": "application/json"}
        if hasattr(servico, "headers"):
            self.headers = servico.headers
        self.processar_resposta = processar_resposta

    def define_url_com_autenticacao(self, atualiza_autenticacao=False):
        requisicao_url = self.servico.url
        if self.servico.exige_autenticacao:
            credenciais = Credenciais(self.meio_pagamento, self.configuracao_pagamento)
            autenticacao = credenciais.define_autenticacao(requisicao_url, atualiza_autenticacao)
            if credenciais.credenciador.tipo == TipoAutenticacao.cabecalho_http:
                self.headers.update(autenticacao)
            if credenciais.credenciador.tipo == TipoAutenticacao.query_string:
                return autenticacao
        return requisicao_url

    def request(self, metodo=TipoMetodo.post, segunda_tentativa=False):
        url = self.define_url_com_autenticacao(atualiza_autenticacao=segunda_tentativa)
        dados_envio = {}
        dados = None
        if hasattr(self.servico, "dados"):
            dados = self.servico.dados
            dados_envio = dados
        if self.servico.envio_por_querystring:
            resposta = getattr(requests, metodo)(url, params=dados_envio, headers=self.headers)
        else:
            if type(dados is dict):
                dados_envio = json.dumps(dados)
            resposta = getattr(requests, metodo)(url, data=dados_envio, headers=self.headers)
        if self.processar_resposta:
            resposta = self.processar_resposta(resposta)
        if settings.GRAVA_EVIDENCIA and self.tipo == TipoConexao.envio:
            grava_evidencia_envio(self.meio_pagamento, para=url, dados=dados, resposta=resposta)
        return resposta

    def get(self, segunda_tentativa=False):
        return self.request(TipoMetodo.get, segunda_tentativa)

    def post(self, segunda_tentativa=False):
        return self.request(TipoMetodo.post, segunda_tentativa)
