# -*- coding: utf-8 -*-
from pagador.acesso.externo import Conexao
from pagador.pedido import notificador
from pagador.envio.models import Pedido
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from pagador.retorno.models import SituacaoPedido
from pagador.seguranca.autenticador import TipoConexao
from pagador.utils import Formatador


class Enviar(object):
    def __init__(self, pedido, dados, configuracao_pagamento):
        self.pedido = pedido
        self.conta_id = None
        self.pedido_numero = None
        if pedido:
            self.conta_id = self.pedido.conta_id
            self.pedido_numero = self.pedido.numero
        self.dados = dados
        self.configuracao_pagamento = configuracao_pagamento

    @property
    def formatador(self):
        return Formatador

    def gerar_dados_de_envio(self):
        return self.dados

    def obter_situacao_do_pedido(self, status_requisicao):
        if status_requisicao == 200:
            return SituacaoPedido.SITUACAO_PEDIDO_PAGO
        return SituacaoPedido.SITUACAO_PEDIDO_CANCELADO

    #FIXME: rever isso ao refatorar o sistema de pedido
    @property
    def valor_envio(self):
        return self.pedido.valor_total - self.pedido.valor_subtotal


class EnviarPedido(Plugavel):
    def __init__(self, pedido_numero, dados, conta_id, configuracao_pagamento):
        self.meio_pagamento = configuracao_pagamento.meio_pagamento
        self.pedido = Pedido.objects.get(numero=pedido_numero, conta_id=conta_id)
        self.dados = dados
        self.configuracao_pagamento = configuracao_pagamento
        self.headers = {"Content-Type": "application/json"}
        self._requisicao = None

    def processar_resposta(self, resposta=None):
        if getattr(self.requisicao, "processa_resposta", False):
            return self.requisicao.processar_resposta(resposta)
        return {"data": resposta.content, "status": resposta.status_code, "reenviar": False}

    @property
    def requisicao(self):
        if not self._requisicao:
            try:
                self._requisicao = self.plugin_requisicao.EnviarPedido(self.pedido, self.dados, self.configuracao_pagamento)
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe requisicao.EnviarPedido.".format(self.meio_pagamento))
        return self._requisicao

    def post(self, segunda_tentativa=False):
        conexao = Conexao(self.meio_pagamento, self.configuracao_pagamento, self.requisicao, TipoConexao.envio, self.processar_resposta)
        return conexao.post(segunda_tentativa)

    def enviar(self):
        if self.requisicao.url:
            resultado = self.post()
        else:
            resultado = self.processar_resposta()
        if resultado.get("reenviar", False):
            resultado = self.post(segunda_tentativa=True)
        if resultado["status"] == 201 or resultado["status"] == 200:
            if self.requisicao.grava_identificador:
                self.grava_identificacao_de_pagamento(resultado["identificador"])
            situacao_do_pedido = self.requisicao.obter_situacao_do_pedido(resultado["status"])
            if situacao_do_pedido:
                notificador.envia_alteracao_de_situacao_de_pedido(self.pedido.numero, situacao_do_pedido, self.pedido.conta_id)
        return resultado

    def grava_identificacao_de_pagamento(self, identificacao):
        pedido_pagamento = self.pedido.pedido_pagamentos.get(pagamento=self.pedido.pagamento)
        pedido_pagamento.identificador_id = identificacao
        pedido_pagamento.save()
