# -*- coding: utf-8 -*-
import json
import traceback
from django.db import IntegrityError
from flask import request, logging, redirect

from flask.ext import restful
from werkzeug.exceptions import BadRequest

from pagador import MeioDePagamentoNaoEncontrado, MeioDePagamentoNaoDefinido
from pagador.configuracao.cadastro import DadosInvalidos
from pagador.configuracao.models import FormaDePagamento, FormaDePagamentoConfiguracao
from pagador.envio.requisicao import EnviarPedido
from pagador.extensao.base import PluginNaoEncontrado
from pagador.retorno.registro import RegistrarPagamento
from api_pagador.app import autenticacao
from pagador.seguranca.instalacao import InstalacaoNaoFinalizada

logger = logging.getLogger("API-PAGADOR")


class HelloWorld(restful.Resource):
    def get(self):
        return {"hello": "world pagador"}


class FormasPagamentoResource(restful.Resource):
    @autenticacao.requerido
    def get(self):
        conta_id = request.args.get('conta_id', None)
        plano_indice = request.args.get('plano_indice', 1)
        soh_habilitados = request.args.get('soh_habilitados', "False") == "True"
        com_configuracao = False
        if conta_id:
            com_configuracao = True
            formas_pagamento = FormaDePagamento.configuracoes_ativas(conta_id=conta_id)
        else:
            formas_pagamento = FormaDePagamento.objects.filter(ativo=True)

        formas_dict = []
        for forma_pagamento in formas_pagamento:
            forma_dict = forma_pagamento.to_dict(com_configuracao=com_configuracao, plano_indice=plano_indice, soh_habilitados=soh_habilitados)
            if forma_dict:
                formas_dict.append(forma_dict)
        return {"formas_pagamento": formas_dict}


class PagamentoBaseResource(restful.Resource):
    def __init__(self):
        self.pagamento_id = None
        self.conta_id = None
        self._forma_de_pagamento = None
        self._dados = None

    @property
    def dados(self):
        if not self._dados:
            self._dados = {}
            for arg in request.args:
                self._dados[arg] = request.args[arg]
            for arg in request.form:
                self._dados[arg] = request.form[arg]
        return self._dados

    def retorno(self, pagamento_id=None):
        if pagamento_id:
            self.pagamento_id = pagamento_id
        if self.retorno_tem_erro:
            status = self.forma_de_pagamento_to_dict["status"]
            del self.forma_de_pagamento_to_dict["status"]
            return self.forma_de_pagamento_to_dict, status
        return self.forma_de_pagamento_to_dict, 200

    @property
    def retorno_tem_erro(self):
        return "erro" in self.forma_de_pagamento_to_dict

    @property
    def forma_de_pagamento(self):
        if not self._forma_de_pagamento:
            try:
                self._forma_de_pagamento = FormaDePagamento.objects.get(id=self.pagamento_id, ativo=True)
            except FormaDePagamento.DoesNotExist:
                self._forma_de_pagamento = None
        return self._forma_de_pagamento

    @property
    def forma_de_pagamento_to_dict(self):
        if not self._forma_de_pagamento:
            try:
                self._forma_de_pagamento = FormaDePagamento.objects.get(id=self.pagamento_id, ativo=True)
                return self._forma_de_pagamento.to_dict(com_configuracao=False)
            except FormaDePagamento.DoesNotExist:
                return {"erro": u"Não foi encontrada forma de pagamento ativa com id {}".format(self.pagamento_id), "status": 404}
        return self._forma_de_pagamento.to_dict(com_configuracao=False)


class FormaPagamentoResource(PagamentoBaseResource):
    @autenticacao.requerido
    def get(self, pagamento_id):
        return self.retorno(pagamento_id)


class PagamentoConfiguracaoBaseResource(restful.Resource):

    def __init__(self):
        self._dados = None
        self._pagamento_configuracao = None
        self.pagamento_id = None
        self.conta_id = None

    @property
    def dados(self):
        if not self._dados:
            self._dados = {}
            for arg in request.args:
                self._dados[arg] = request.args[arg]
            for arg in request.form:
                self._dados[arg] = request.form[arg]
            try:
                self._dados.update(request.get_json())
            except TypeError:
                pass
            except BadRequest:
                pass
        return self._dados

    @property
    def chave(self):
        raise NotImplementedError

    @property
    def valor(self):
        raise NotImplementedError

    @property
    def pagamento_configuracao(self):
        if self._pagamento_configuracao is None:
            if self.conta_id:
                self._pagamento_configuracao = FormaDePagamentoConfiguracao.objects.get(
                    conta_id=self.conta_id,
                    forma_pagamento_id=self.pagamento_id
                )
        return self._pagamento_configuracao

    @autenticacao.requerido
    def get(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        try:
            return {self.chave: self.valor}
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            logger.error(u"Erro de plugin em {}: {}".format(self.chave, ex))
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            logger.error(u"Erro de banco em {}: {}".format(self.chave, ex))
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500
        except InstalacaoNaoFinalizada, ex:
            logger.error(u"Erro de instalação {}: {}".format(self.chave, ex))
            return {"erro": u"Erro de instalação: {}".format(ex)}, 500


class FormaPagamentoConfiguracaoResource(restful.Resource):
    @autenticacao.requerido
    def get(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        try:
            pagamento_configuracao, criado = FormaDePagamentoConfiguracao.objects.get_or_create(
                conta_id=conta_id,
                forma_pagamento_id=pagamento_id
            )
            return pagamento_configuracao.to_dict()
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            logger.error(u"Erro de plugin em {}: {}".format("pagamento configuracao", ex))
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            logger.error(u"Erro de banco em {}: {}".format("pagamento configuracao", ex))
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500

    #FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
    @autenticacao.requerido
    def post(self, pagamento_id, conta_id):
        dados = request.get_json()
        try:
            pagamento_configuracao, criado = FormaDePagamentoConfiguracao.objects.get_or_create(
                conta_id=conta_id,
                forma_pagamento_id=pagamento_id,
            )
            pagamento_configuracao.salvar(dados)
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500
        except DadosInvalidos, ex:
            return {"validacao": u"{}".format(ex), "erros": ex.erros}, 400


#FIXME: Refatorar isso, pois tá repetido
class FormaPagamentoConfiguracaoComplementoResource(PagamentoConfiguracaoBaseResource):
    #FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
    @autenticacao.requerido
    def get(self, pagamento_id, conta_id):
        return self.post(pagamento_id, conta_id)

    #FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
    @autenticacao.requerido
    def post(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        try:
            resultado = self.pagamento_configuracao.complemento(self.dados)
            return {"message": resultado["content"]}, resultado["status"]
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoScriptResource(PagamentoConfiguracaoBaseResource):

    @property
    def chave(self):
        return "scripts"

    @property
    def valor(self):
        return self.pagamento_configuracao.script.to_dict()


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoSelecaoResource(PagamentoConfiguracaoBaseResource):
    @property
    def chave(self):
        return "scripts"

    @property
    def valor(self):
        return self.pagamento_configuracao.selecao.to_dict()


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoIntegracaoResource(PagamentoConfiguracaoBaseResource):
    @property
    def chave(self):
        return "aplicacao"

    @property
    def valor(self):
        return {"url": self.pagamento_configuracao.instalador.url_ativador(self.dados)}

    @autenticacao.requerido
    def delete(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        try:
            resultado = self.pagamento_configuracao.desinstalar()
            if resultado:
                return resultado
            return {"resultado": "OK"}, 200
        except Exception, ex:
            return {"erro": u"Ocorreu um erro durante a remoção do aplicativo. Por favor, tente novamente."}, 500
            #PARA DEBUG
            # stack = traceback.format_exc()
            # return {"erro": u"Servidor: {}\n\n{}".format(ex, stack)}, 500


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoInstalacaoResource(PagamentoConfiguracaoBaseResource):
    def get(self, pagamento_id, conta_id):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        mensagem = None
        next_url = self.dados["next_url"]
        try:
            self.pagamento_configuracao.instalar(self.dados)
        except Exception, ex:
            mensagem = u"Erro: {}".format(ex)
        if mensagem:
            if '?' in next_url:
                return redirect(u"{}&mensagem={}".format(next_url, mensagem), code=301)
            return redirect(u"{}?mensagem={}".format(next_url, mensagem), code=301)
        return redirect(next_url, code=301)


#FIXME: ALERTA!!! ALERTA!!! SEM TESTES!!! ALERTA!!! ALERTA!!!
class FormaPagamentoConfiguracaoEnviarResource(PagamentoConfiguracaoBaseResource):
    @autenticacao.requerido
    def post(self, pagamento_id, conta_id, pedido_numero):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        try:
            enviar_pedido = EnviarPedido(pedido_numero, self.dados, conta_id, self.pagamento_configuracao)
            resultado = enviar_pedido.enviar()
            return resultado
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500
        except Exception, ex:
            import traceback
            return {"erro": u"Servidor: {} - {}".format(ex, traceback.format_exc())}, 500


class FormaPagamentoRetornoResource(PagamentoConfiguracaoBaseResource):
    def post(self, pagamento_id, conta_id=None, tipo="retorno"):
        return self.get(pagamento_id, conta_id, tipo)

    def get(self, pagamento_id, conta_id=None, tipo="retorno"):
        self.pagamento_id = pagamento_id
        self.conta_id = conta_id
        try:
            self.dados["conta_id"] = conta_id
            if conta_id:
                registro = RegistrarPagamento(self.dados, tipo=tipo, configuracao_pagamento=self.pagamento_configuracao)
            else:
                registro = RegistrarPagamento(self.dados, tipo=tipo, pagamento_id=pagamento_id)
            resultado = registro.processar()
            if "redirect" in resultado:
                return redirect(resultado["redirect"], code=301)
            return resultado
        except (MeioDePagamentoNaoDefinido, MeioDePagamentoNaoEncontrado, PluginNaoEncontrado), ex:
            return {"erro": u"pagador: {}".format(ex)}, 500
        except IntegrityError, ex:
            return {"erro": u"Erro no banco de dados: {}".format(ex)}, 500
        # except Exception, ex:
            # return {"erro": u"Servidor: {}".format(ex)}, 500
