# -*- coding: utf-8 -*-
import json
import requests
from pagador import settings
from pagador.envio.notificador import grava_evidencia_envio
from pagador.seguranca.autenticador import Credenciais, TipoAutenticacao, TipoMetodo, TipoConexao


class FormatoDeEnvio(object):
    querystring = 1
    json = 2
    form_urlencode = 3


class Conexao(object):
    def __init__(self, meio_pagamento, configuracao_pagamento, servico, tipo="envio", processar_resposta=None):
        self.tipo = tipo
        self.meio_pagamento = meio_pagamento
        self.configuracao_pagamento = configuracao_pagamento
        self.servico = servico
        self.headers = {"Content-Type": "application/json"}
        self.dados_envio = {}
        if hasattr(servico, "headers"):
            self.headers = servico.headers
        self.processar_resposta = processar_resposta

    def define_url_com_autenticacao(self, atualiza_autenticacao=False):
        requisicao_url = self.servico.url
        if self.servico.exige_autenticacao:
            credenciais = Credenciais(self.meio_pagamento, self.configuracao_pagamento)
            autenticacao = credenciais.define_autenticacao(requisicao_url, atualiza_autenticacao)
            if credenciais.credenciador.tipo == TipoAutenticacao.cabecalho_http:
                self.headers.update(autenticacao)
            if credenciais.credenciador.tipo == TipoAutenticacao.form_urlencode:
                self.dados_envio.update(autenticacao)
            if credenciais.credenciador.tipo == TipoAutenticacao.query_string:
                return autenticacao
        return requisicao_url

    def request(self, metodo=TipoMetodo.post, segunda_tentativa=False):
        url = self.define_url_com_autenticacao(atualiza_autenticacao=segunda_tentativa)
        dados = self.servico.gerar_dados_de_envio()
        if dados:
            self.dados_envio.update(dados)
        if self.servico.formato_de_envio == FormatoDeEnvio.querystring:
            resposta = getattr(requests, metodo)(url, params=self.dados_envio, headers=self.headers)
        elif self.servico.formato_de_envio == FormatoDeEnvio.form_urlencode:
            resposta = getattr(requests, metodo)(url, data=self.dados_envio, headers=self.headers)
        else:
            dados_envio = self.dados_envio
            if type(self.dados_envio is dict):
                dados_envio = json.dumps(self.dados_envio)
            resposta = getattr(requests, metodo)(url, data=dados_envio, headers=self.headers)
        if self.processar_resposta:
            resposta = self.processar_resposta(resposta)
        if settings.GRAVA_EVIDENCIA and self.tipo == TipoConexao.envio:
            grava_evidencia_envio(self.meio_pagamento, para=url, dados=self.dados_envio, resposta=resposta, conta_id=self.servico.conta_id, pedido_numero=self.servico.pedido_numero)
        return resposta

    def get(self, segunda_tentativa=False):
        return self.request(TipoMetodo.get, segunda_tentativa)

    def post(self, segunda_tentativa=False):
        return self.request(TipoMetodo.post, segunda_tentativa)