# -*- coding: utf-8 -*-
from decimal import Decimal
from pagador.acesso.externo import Conexao
from pagador.pedido import notificador
from pagador.envio.models import Pedido
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from pagador.retorno.models import SituacaoPedido
from pagador.seguranca.autenticador import TipoConexao
from pagador.utils import Formatador


class Enviar(object):
    def __init__(self, pedido, dados, configuracao_pagamento):
        self.pedido = pedido
        self.conta_id = None
        self.pedido_numero = None
        if pedido:
            self.conta_id = self.pedido.conta_id
            self.pedido_numero = self.pedido.numero
        self.dados = dados
        self.configuracao_pagamento = configuracao_pagamento
        self.passo_atual = None
        self.processa_resposta = False

    @property
    def formatador(self):
        return Formatador

    def gerar_dados_de_envio(self, passo=None):
        return self.dados

    def obter_situacao_do_pedido(self, status_requisicao):
        if status_requisicao == 200:
            return SituacaoPedido.SITUACAO_PEDIDO_PAGO
        return SituacaoPedido.SITUACAO_PEDIDO_CANCELADO

    @property
    def valores_de_pagamento(self):
        return None

    #FIXME: rever isso ao refatorar o sistema de pedido
    @property
    def valor_envio(self):
        valor_envio = self.pedido.valor_envio or Decimal("0.00")
        valor_desconto = self.pedido.valor_desconto or Decimal("0.00")
        if valor_envio == valor_desconto:
            return Decimal("0.00")
        return valor_envio

    @property
    def valor_desconto(self):
        valor_envio = self.pedido.valor_envio or Decimal("0.00")
        valor_desconto = self.pedido.valor_desconto or Decimal("0.00")
        if valor_envio == valor_desconto:
            return Decimal("0.00")
        return valor_desconto

    @property
    def valor_subtotal(self):
        valor_subtotal = self.pedido.valor_subtotal or Decimal("0.00")
        return valor_subtotal - self.valor_desconto

    @property
    def nome_do_cliente(self):
        nome = self.pedido.cliente.nome
        nome = self.formatador.trata_nome(nome)
        return self.formatador.trata_unicode_com_limite(nome, limite=50, ascii=True, trata_espaco_duplo=True)

    @property
    def telefone_do_cliente(self):
        telefone = None
        if self.pedido.cliente.telefone_principal:
            telefone = self.pedido.cliente.telefone_principal
        elif self.pedido.cliente.telefone_comercial:
            telefone = self.pedido.cliente.telefone_comercial
        elif self.pedido.cliente.telefone_celular:
            telefone = self.pedido.cliente.telefone_celular
        if telefone:
            return self.formatador.converte_tel_em_tupla_com_ddd(telefone)
        return '', ''

    @property
    def documento_de_cliente(self):
        if self.pedido.cliente.endereco.tipo == "PF":
            return self.pedido.cliente.endereco.cpf
        else:
            return self.pedido.cliente.endereco.cnpj

    @property
    def nome_para_entrega(self):
        if self.pedido.endereco_entrega.tipo == "PF":
            return self.formatador.trata_nome(self.pedido.endereco_entrega.nome)
        else:
            return self.formatador.trata_nome(self.pedido.endereco_entrega.razao_social)


class EnviarPedido(Plugavel):
    def __init__(self, pedido_numero, dados, conta_id, configuracao_pagamento):
        self.meio_pagamento = configuracao_pagamento.meio_pagamento
        self.pedido = Pedido.objects.get(numero=pedido_numero, conta_id=conta_id)
        self.dados = dados
        self.configuracao_pagamento = configuracao_pagamento
        self.headers = {"Content-Type": "application/json"}
        self._requisicao = None

    def processar_resposta(self, resposta=None):
        if self.requisicao.processa_resposta:
            return self.requisicao.processar_resposta(resposta)
        return {"data": resposta.content, "status": resposta.status_code, "reenviar": False}

    @property
    def requisicao(self):
        if not self._requisicao:
            try:
                self._requisicao = self.plugin_requisicao.EnviarPedido(self.pedido, self.dados, self.configuracao_pagamento)
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe requisicao.EnviarPedido.".format(self.meio_pagamento))
        return self._requisicao

    def post(self, segunda_tentativa=False, passo=None):
        conexao = Conexao(self.meio_pagamento, self.configuracao_pagamento, self.requisicao, TipoConexao.envio, self.processar_resposta)
        return conexao.post(segunda_tentativa, passo)

    def enviar(self):
        if not self.configuracao_pagamento:
            return {"content": {"mensagem": u"A forma de pagamento selecionada não está configurada na loja.", "fatal": True}, "status": 404, "reenviar": False}
        if not self.configuracao_pagamento.forma_de_pagamento["configurado"] or not self.configuracao_pagamento.ativo:
            return {"content": {"mensagem": u"A forma de pagamento selecionada não está configurada na loja.", "fatal": True}, "status": 404, "reenviar": False}
        if self.requisicao.url:
            resultado = self.post()
        else:
            resultado = self.processar_resposta()
        if resultado.get("reenviar", False):
            resultado = self.post(segunda_tentativa=True)
        if resultado["status"] == 201 or resultado["status"] == 200:
            if self.requisicao.deve_gravar_dados_de_pagamento:
                identificador_id = resultado.get("identificador", None)
                self.grava_dados_de_pagamento(identificador_id=identificador_id, valores=self.requisicao.valores_de_pagamento)
            situacao_do_pedido = self.requisicao.obter_situacao_do_pedido(resultado["status"])
            if situacao_do_pedido:
                notificador.envia_alteracao_de_situacao_de_pedido(self.pedido.numero, situacao_do_pedido, self.pedido.conta_id)
        return resultado

    def grava_dados_de_pagamento(self, identificador_id=None, valores=None):
        pedido_pagamento = self.pedido.pedido_pagamentos.get(pagamento=self.pedido.pagamento)
        if valores:
            for chave in valores.keys():
                setattr(pedido_pagamento, chave, valores[chave])
        else:
            pedido_pagamento.identificador_id = identificador_id
        pedido_pagamento.save()
