# -*- coding: utf-8 -*-

from pagador.configuracao.cadastro import ValoresBase
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from repositories.configuracao import models
from repositories.configuracao.models import PagamentoBanco


class FormaDePagamento(models.FormaPagamento, Plugavel):
    QUERY_CONFIGURACOES_ATIVAS = """SELECT
                pg.pagamento_id,
                pg.pagamento_nome,
                pg.pagamento_codigo,
                pg.pagamento_ativado,
                pg.pagamento_plano_indice,
                pg.pagamento_parcela_valor_minimo_parcela,
                pgc.pagamento_configuracao_id,
                pgc.pagamento_configuracao_usuario,
                pgc.pagamento_configuracao_senha,
                pgc.pagamento_configuracao_token,
                pgc.pagamento_configuracao_assinatura,
                pgc.pagamento_configuracao_codigo_autorizacao,
                pgc.pagamento_configuracao_ativo,
                pgc.pagamento_configuracao_json,
                pgc.pagamento_configuracao_desconto,
                pgc.pagamento_configuracao_desconto_tipo,
                pgc.pagamento_configuracao_desconto_valor,
                pgc.pagamento_configuracao_quantidade_parcela_maxima,
                pgc.pagamento_configuracao_quantidade_parcela_sem_juros,
                pgc.pagamento_configuracao_juros_valor,
                pgc.pagamento_configuracao_valor_minimo_parcela,
                pgc.conta_id
           FROM configuracao.tb_pagamento as pg
           LEFT OUTER JOIN configuracao.tb_pagamento_configuracao as pgc on (pgc.pagamento_id = pg.pagamento_id AND pgc.conta_id=%s)
           WHERE pg.pagamento_ativado = true{};"""

    class Meta:
        proxy = True

    @property
    def meio_pagamento(self):
        return self.codigo

    def save(self, *args, **kwargs):
        if self.conta and not self.contrato_id:
            self.contrato_id = self.conta.contrato_id
        super(FormaDePagamento, self).save(*args, **kwargs)

    @property
    def imagem_admin(self):
        return 'img/formas-de-pagamento/{}-logo.png'.format(self.codigo)

    @property
    def imagem_loja(self):
        return 'img/bandeiras/{}-logo.png'.format(self.codigo)

    @property
    def bancos_ativos_na_conta(self):
        return [{"id": banco.id, "nome": banco.banco.nome, "imagem": banco.banco.imagem, "codigo": banco.banco.codigo} for banco in self.pagamento_bancos if banco.ativo]

    @property
    def pagamento_bancos(self):
        if not hasattr(self, "_pagamento_bancos"):
            self._pagamento_bancos = None
        if self._pagamento_bancos is None:
            self._pagamento_bancos = [banco for banco in PagamentoBanco.objects.prefetch_related("banco").filter(pagamento_id=self.id, conta_id=self.conta_id)]
        return self._pagamento_bancos

    @property
    def configurado(self):
        try:
            try:
                bancos = self.pagamento_bancos[0]
            except IndexError:
                bancos = None
            return (self.pagamento_configuracao_usuario or
                    self.pagamento_configuracao_senha or
                    self.pagamento_configuracao_assinatura or
                    self.pagamento_configuracao_token or
                    self.pagamento_configuracao_codigo_autorizacao or
                    bancos or self.pagamento_configuracao_json)
        except AttributeError:
            return None

    @classmethod
    def configuracoes_ativas(cls, conta_id, pagamento_id=None):
        parametros = [conta_id]
        if pagamento_id:
            parametros.append(pagamento_id)
        return cls.objects.raw(
            cls.QUERY_CONFIGURACOES_ATIVAS.format((" AND pg.pagamento_id=%s" if pagamento_id else '')), parametros
        )

    @property
    def valor_minimo_da_parcela(self):
        valor = getattr(self, "pagamento_configuracao_valor_minimo_parcela", 0.0)
        if valor > 0:
            return float(valor)
        return float(self.valor_minimo_parcela or 0.0)

    def to_dict(self, com_configuracao=True, plano_indice=None, soh_habilitados=False):
        forma_pagamento = {
            "id": self.id,
            "codigo": self.codigo,
            "nome": self.nome,
            "ativo_no_sistema": self.ativo,
            "imagem_admin": self.imagem_admin,
            "imagem_loja": self.imagem_loja,
            "plano_indice": self.plano_indice,
            "parcelas": [{"fator": float(parcela.fator), "numero_parcelas": parcela.numero_parcelas} for parcela in self.parcelas.all()],
            "valor_minimo_parcela": self.valor_minimo_da_parcela
        }
        if com_configuracao:
            if self.pagamento_configuracao_desconto:
                forma_pagamento["desconto"] = self.pagamento_configuracao_desconto
                forma_pagamento["desconto_tipo"] = self.pagamento_configuracao_desconto_tipo
                forma_pagamento["desconto_valor"] = '{0:.2f}'.format(self.pagamento_configuracao_desconto_valor)
            forma_pagamento["ativo_para_a_conta"] = self.pagamento_configuracao_ativo
            forma_pagamento["configurado"] = self.configurado is not None
            forma_pagamento["maximo_parcelas"] = self.pagamento_configuracao_quantidade_parcela_maxima
            forma_pagamento["parcelas_sem_juros"] = self.pagamento_configuracao_quantidade_parcela_sem_juros
            juros_valor = self.pagamento_configuracao_juros_valor
            if not juros_valor:
                juros_valor = 0.0
            forma_pagamento["juros_valor"] = float(juros_valor)
            if plano_indice:
                try:
                    plano_indice = int(plano_indice)
                except ValueError:
                    plano_indice = 0
                forma_pagamento["habilitado_na_loja"] = forma_pagamento["configurado"] and self.ativo and self.pagamento_configuracao_ativo and self.plano_indice <= plano_indice
            forma_pagamento["bancos_ativos_na_conta"] = self.bancos_ativos_na_conta
        if soh_habilitados and "habilitado_na_loja" in forma_pagamento and not forma_pagamento["habilitado_na_loja"]:
            return None
        return forma_pagamento


class PagamentoNaoConfigurado(Exception):
    pass


class FormaDePagamentoConfiguracao(models.FormaPagamentoConfiguracao, Plugavel):
    class Meta:
        proxy = True

    @property
    def meio_pagamento(self):
        if "codigo" in self.forma_de_pagamento:
            return self.forma_de_pagamento["codigo"]
        raise PagamentoNaoConfigurado("A forma de pagamento_id {} para a conta_id {} não foi configurada corretamente. Verifique com o suporte.".format(self.forma_pagamento_id, self.conta_id))

    @property
    def forma_de_pagamento(self):
        if not hasattr(self, "_forma_de_pagamento"):
            self._forma_de_pagamento = None
        if self._forma_de_pagamento is None:
            try:
                self._forma_de_pagamento = FormaDePagamento.configuracoes_ativas(self.conta_id, pagamento_id=self.forma_pagamento_id)[0].to_dict(com_configuracao=True)
            except IndexError:
                self._forma_de_pagamento = {}
        return self._forma_de_pagamento

    def bancos_configurados_na_conta(self, banco_id=None):
        if not hasattr(self, "_bancos_configurados"):
            self._bancos_configurados = None
        if self._bancos_configurados is None:
            if banco_id:
                self._bancos_configurados = models.PagamentoBanco.objects.get_or_create(conta_id=self.conta_id, pagamento_id=self.forma_pagamento_id, banco_id=banco_id)
            else:
                self._bancos_configurados = models.PagamentoBanco.objects.filter(conta_id=self.conta_id, pagamento_id=self.forma_pagamento_id)
        return self._bancos_configurados

    @property
    def bancos(self):
        if not hasattr(self, "_bancos"):
            self._bancos = None
        if self._bancos is None:
            self._bancos = models.Banco.objects.all()
            for banco in self._bancos:
                for pagamento_banco in self.bancos_configurados_na_conta():
                    if banco.id == pagamento_banco.banco_id:
                        banco.pagamento_banco = pagamento_banco
            return self._bancos

    def save(self, *args, **kwargs):
        self.desconto_tipo = self.TIPO_PORCENTAGEM
        self.desconto = self.desconto_valor and True or False
        if self.maximo_parcelas and self.parcelas_sem_juros:
            self.parcelas_sem_juros = int(self.parcelas_sem_juros)
            self.maximo_parcelas = int(self.maximo_parcelas)
            if self.maximo_parcelas != 0 and self.parcelas_sem_juros > self.maximo_parcelas:
                self.parcelas_sem_juros = self.maximo_parcelas
        if self.conta and not self.contrato_id:
            self.contrato_id = self.conta.contrato_id

        if self.email_comprovante and self.conta_id and self.conta.email == self.email_comprovante:
            self.email_comprovante = None

        super(FormaDePagamentoConfiguracao, self).save(*args, **kwargs)

    @property
    def formulario(self):
        if not hasattr(self, "_formulario"):
            self._formulario = None
        if not self._formulario:
            try:
                self._formulario = self.plugin_configuracao.Formulario()
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento {} (id={}) não possui a classe configuracao.Formulario.".format(self.meio_pagamento, self.forma_pagamento_id))
        return self._formulario

    @property
    def script(self):
        try:
            return self.plugin_configuracao.MeioPagamentoEnvio()
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoEnvio.".format(self.meio_pagamento))

    def selecao(self, dados=None):
        try:
            return self.plugin_configuracao.MeioPagamentoSelecao(self, dados)
        except AttributeError, ex:
            if "NoneType" in unicode(ex):
                raise
            if "MeioPagamentoSelecao" in unicode(ex):
                raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoSelecao.".format(self.meio_pagamento))
            raise

    def instalar(self, dados):
        dados_instalacao = self.instalador.dados_de_instalacao(dados)
        for campo in dados_instalacao.keys():
            setattr(self, campo, dados_instalacao[campo])
        self.save()

    def dados_para_instalador(self):
        dados = {}
        for campo in self.instalador.campos:
            dados[campo] = getattr(self, campo, None)
        return dados

    def atualizar(self):
        dados = self.dados_para_instalador()
        dados_atualizacao = self.instalador.dados_de_atualizacao(dados)
        for campo in dados_atualizacao.keys():
            setattr(self, campo, dados_atualizacao[campo])
        self.save()

    def desinstalar(self):
        dados = self.dados_para_instalador()
        for campo in dados:
            setattr(self, campo, None)
        resultado = self.instalador.desinstalar(dados)
        self.save()
        return resultado

    @property
    def instalador(self):
        try:
            return self.plugin_seguranca.Instalador(self, id=self.conta_id)
        except AttributeError, ex:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe seguranca.Instalador.".format(self.meio_pagamento))

    @property
    def valores(self):
        try:
            return ValoresBase(self, self.formulario)
        except AttributeError:
            raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoValores.".format(self.meio_pagamento))

    @property
    def cadastro(self):
        try:
            return self.plugin_configuracao.MeioPagamentoCadastro(self)
        except AttributeError:
            return None
            #FIXME: Pensar em um modelo melhor para determinar extensoes não obrigatórias
            # raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe configuracao.MeioPagamentoCadastro.".format(self.meio_pagamento))

    @property
    def eh_aplicacao(self):
        try:
            return self.plugin_configuracao.eh_aplicacao
        except AttributeError:
            return False
            #FIXME: Pensar em um modelo melhor para determinar extensoes não obrigatórias

    def salvar(self, dados):
        self.formulario.define_valores_em_model(self, dados).save()

    def complemento(self, dados):
        return self.cadastro.complemento(dados)

    def to_dict(self):
        cadastro = None
        if self.cadastro:
            cadastro = self.cadastro.to_dict()
        return {
            "cadastro": cadastro,
            "valores": self.valores.to_dict(),
            "formulario": self.formulario.to_dict(),
            "pagamento": self.forma_de_pagamento,
            "eh_aplicacao": self.eh_aplicacao
        }


class Banco(models.Banco):
    class Meta:
        proxy = True


class BoletoCarteira(models.BoletoCarteira):
    class Meta:
        proxy = True
