# -*- coding: utf-8 -*-
import json
import os
from importlib import import_module

from pagador import settings
from pagador.utils import Formatador


class TemplateNaoEncontrado(Exception):
    pass


def caminho_para_template(arquivo, meio_pagamento=None, base_para_diretorio=None):
    diretorios = getattr(settings, "DIRETORIOS_DOS_TEMPLATES", {})
    if not diretorios:
        if meio_pagamento:
            import_meio_pagamento = settings.LIBS_MEIOS_DE_PAGAMENTO[meio_pagamento]
            modulo = import_module("{}.{}".format(import_meio_pagamento, "configuracao"))
            diretorio = os.path.dirname(os.path.realpath(modulo.__file__))
            return os.path.join(diretorio, "templates", arquivo)
        if base_para_diretorio:
            diretorio = os.path.dirname(os.path.realpath(base_para_diretorio))
            return os.path.join(diretorio, "templates", arquivo)
    if meio_pagamento and meio_pagamento in diretorios:
        diretorio = diretorios[meio_pagamento]
        return os.path.join(diretorio, arquivo)
    for chave in diretorios:
        caminho = os.path.join(diretorios[chave], arquivo)
        if os.path.exists(caminho):
            return caminho
    raise TemplateNaoEncontrado("O template para o arquivo {} não foi encontrado.". format(arquivo))


class TipoDeCampo(object):
    texto = "CharField"
    area = "TextArea"
    boleano = "BooleanField"
    escolha = "ChoiceField"
    decimal = "BRDecimalField"
    oculto = "Hidden"


class FormatoDeCampo(object):
    texto = "texto"
    ascii = "ascii"
    html = "html"
    xml = "xml"
    json = "json"
    numero = "numero"


class ValidadorBase(object):
    def __init__(self, valor):
        self.valor = valor
        self.erros = {}

    @property
    def eh_valido(self):
        return True


class CampoFormulario(object):
    def __init__(self, nome, label, bound=True, descricao=None, requerido=False, tipo=TipoDeCampo.texto, formato=FormatoDeCampo.texto, invisivel=False,
                 tamanho_min=None, tamanho_max=None, ordem=0, texto_ajuda=None, opcoes=(), grupo=None, valor_padrao=None, validador=None, decimais=2):
        self.bound = bound
        self.ordem = ordem
        self.nome = nome
        self.label = label
        self.descricao = descricao
        self.requerido = requerido
        self.tipo = tipo
        self.formato = formato
        self.invisivel = invisivel
        self.tamanho_min = tamanho_min
        self.tamanho_max = tamanho_max
        self.texto_ajuda = texto_ajuda
        self.opcoes = opcoes
        self.grupo = grupo
        self.valor_padrao = valor_padrao
        self.validador = validador
        self.decimais = decimais

    def to_dict(self):
        return {
            "nome": self.nome,
            "ordem": self.ordem,
            "label": self.label,
            "bound": self.bound,
            "descricao": self.descricao,
            "requerido": self.requerido,
            "tipo": self.tipo,
            "invisivel": self.invisivel,
            "tamanho_min": self.tamanho_min,
            "tamanho_max": self.tamanho_max,
            "texto_ajuda": self.texto_ajuda,
            "opcoes": self.opcoes,
            "grupo": self.grupo,
            "valor_padrao": self.valor_padrao,
            "decimais": self.decimais,
        }


class DadosInvalidos(Exception):
    def __init__(self, msg, erros):
        super(DadosInvalidos, self).__init__(msg)
        self.erros = erros


class FormularioBase(object):
    def __init__(self):
        self._fields = [field for field in dir(self) if field != "fields" and isinstance(getattr(self, field), CampoFormulario)]

    @property
    def fields(self):
        return {field: getattr(self, field) for field in self._fields}

    def to_dict(self):
        return {field: self.fields[field].to_dict() for field in self.fields}

    def validar_valores(self, valores):
        erros = {}
        for field in self.fields:
            field = self.fields[field]
            valor = valores.get(field.nome, None)
            if field.validador:
                validacao = field.validador(valor)
                if not validacao.eh_valido:
                    erros[field.nome] = validacao.erros
        return erros

    def define_valores_em_model(self, model, valores):
        ativo = valores.get("ativo", False)
        if ativo:
            erros = self.validar_valores(valores)
            if erros:
                raise DadosInvalidos(u"Ocorreram erros de validação", erros)
        model.ativo = ativo
        for field in self.fields:
            field = self.fields[field]
            if field.bound:
                valor = valores.get(field.nome, None)
                if valor and field.formato == FormatoDeCampo.json:
                    valor = json.loads(valor)
                if valor and field.formato == FormatoDeCampo.ascii:
                    valor = Formatador.string_para_ascii(valor)
                setattr(model, field.nome, valor)
        return model


class ValoresBase(FormularioBase):
    def __init__(self, model, formulario):
        self.formulario = formulario
        self.model = model
        super(ValoresBase, self).__init__()

    @property
    def fields(self):
        return self.formulario.fields

    def to_dict(self):
        self_dict = {
            'ativo': {
                'campo': 'ativo',
                'valor': self.model.ativo
            }
        }
        self_dict.update({field: {"campo": self.fields[field].nome, "valor": self.obtem_valor_serializavel(field)} for field in self.fields if self.fields[field].bound})
        return self_dict

    def obtem_valor_serializavel(self, field):
        valor = getattr(self.model, self.fields[field].nome)
        if not valor and self.fields[field].valor_padrao:
            valor = self.fields[field].valor_padrao
        if not valor is None:
            if self.fields[field].tipo == TipoDeCampo.decimal:
                valor = '{0:.{1}f}'.format(valor, self.fields[field].decimais)
            if self.fields[field].formato == FormatoDeCampo.json and valor:
                valor = json.dumps(valor)
        return valor


class CadastroBase(object):
    def __init__(self, configuracao):
        self.configuracao = configuracao


class SelecaoBase(object):
    def __init__(self, configuracao, dados=None):
        self.dados = dados
        self.configuracao = configuracao