# -*- coding: utf-8 -*-

from pagador import settings
from pagador.acesso.externo import Conexao, TipoMetodo
from pagador.configuracao.models import FormaDePagamento
from pagador.extensao.base import Plugavel, PluginNaoEncontrado
from pagador.pedido import notificador
from pagador.retorno.models import PedidoPagamento, Pedido
from pagador.retorno.notificador import grava_evidencia_retorno
from pagador.seguranca.autenticador import TipoConexao
from pagador.utils import Formatador


class RegistroBase(object):
    def __init__(self, dados, configuracao):
        self.configuracao = configuracao
        self.dados = dados
        self.exige_autenticacao = False
        self.processa_resposta = False
        self.metodo_request = TipoMetodo.post

    @property
    def formatador(self):
        return Formatador

    @property
    def url(self):
        raise NotImplementedError

    @property
    def pedido_numero(self):
        raise NotImplementedError

    @property
    def identificador_id(self):
        raise NotImplementedError

    @property
    def deve_gravar_dados_de_pagamento(self):
        return False

    @property
    def valores_de_pagamento(self):
        return None

    @property
    def situacao_do_pedido(self):
        raise NotImplementedError

    @property
    def alterar_situacao(self):
        raise NotImplementedError

    @property
    def redireciona_para(self):
        raise NotImplementedError

    def processar_resposta(self, resposta):
        raise NotImplementedError

    def gerar_dados_de_envio(self):
        return self.dados


class RegistrarPagamento(Plugavel):
    def __init__(self, dados, tipo="retorno", configuracao_pagamento=None, pagamento_id=None):
        if configuracao_pagamento:
            self.meio_pagamento = configuracao_pagamento.meio_pagamento
        if pagamento_id:
            self.meio_pagamento = FormaDePagamento.objects.get(id=int(pagamento_id)).codigo
        self.configuracao_pagamento = configuracao_pagamento
        self.dados = dados
        self._registrador = None
        self._conta_id = None
        self.tipo = tipo

    @property
    def registrador(self):
        if not self._registrador:
            try:
                self._registrador = self.plugin_retorno.Registro(self.dados, self.tipo, self.configuracao_pagamento)
            except AttributeError:
                raise PluginNaoEncontrado(u"O meio de pagamento '{}' não possui a classe retorno.Registro.".format(self.meio_pagamento))
        return self._registrador

    @property
    def conta_id(self):
        if self._conta_id:
            return self._conta_id
        if self.configuracao_pagamento:
            self._conta_id = self.configuracao_pagamento.conta_id
        elif self.registrador.identificador_id:
            try:
                pedido_pagamento = PedidoPagamento.objects.get(identificador_id=self.registrador.identificador_id)
                self._conta_id = pedido_pagamento.conta_id
            except PedidoPagamento.DoesNotExist, PedidoPagamento.MultipleObjectsReturned:
                self._conta_id = None
        return self._conta_id

    @property
    def achou_registro(self):
        if not self.conta_id:
            return False
        if not self.registrador.pedido_numero:
            return False
        try:
            Pedido.objects.get(conta_id=self.conta_id, numero=self.registrador.pedido_numero)
            return True
        except Exception:
            return False

    def processar_resposta(self, resposta):
        if getattr(self.registrador, "processa_resposta", False):
            return self.registrador.processar_resposta(resposta)
        return {"data": resposta.content, "status": resposta.status_code, "reenviar": False}

    def processar(self):
        resultado = {}
        if self.registrador.obter_dados_do_gateway:
            self.registrador.dados.update(self.obtem_dados_do_gateway())
        if self.registrador.deve_gravar_dados_de_pagamento:
            self.grava_dados_de_pagamento(valores=self.registrador.valores_de_pagamento)
        if self.registrador.alterar_situacao:
            if self.achou_registro:
                resultado["pedido_venda_situacao_id"] = self.registrador.situacao_do_pedido
                notificador.envia_alteracao_de_situacao_de_pedido(self.registrador.pedido_numero, self.registrador.situacao_do_pedido, self.conta_id)
            else:
                resultado = {"resultado": "erro", "mensagem": u"Nada foi registrado, pois os valores retornados não compõem um registro válido na LI."}
        if settings.GRAVA_EVIDENCIA:
            self.dados["tipo"] = self.tipo
            tipo_evidencia = "retorno" if self.tipo == "retorno" else "redirect"
            enviado = {"status_code": 200}
            enviado.update(resultado)
            grava_evidencia_retorno(self.meio_pagamento, self.dados, self.conta_id, self.registrador.pedido_numero, tipo_evidencia=tipo_evidencia, registrado=self.achou_registro, enviado=enviado)
            resultado["evidencia"] = True
        if self.registrador.redireciona_para:
            return {"redirect": self.registrador.redireciona_para}
        resultado["resultado"] = "OK"
        return resultado

    def request(self, segunda_tentativa=False):
        conexao = Conexao(self.meio_pagamento, self.configuracao_pagamento, self.registrador, TipoConexao.retorno, self.processar_resposta)
        if self.registrador.metodo_request == TipoMetodo.get:
            return conexao.get(segunda_tentativa)
        if self.registrador.metodo_request == TipoMetodo.post:
            return conexao.post(segunda_tentativa)

    def obtem_dados_do_gateway(self):
        resultado = self.request()
        if resultado.get("reenviar", False):
            resultado = self.request(segunda_tentativa=True)
        if resultado["status"] == 200:
            return resultado["content"]
        return {}

    def grava_dados_de_pagamento(self, valores=None):
        if not self.conta_id or not self.registrador.pedido_numero or not self.configuracao_pagamento:
            return None
        pedido = Pedido.objects.get(conta_id=self.conta_id, numero=self.registrador.pedido_numero)
        pedido_pagamento = PedidoPagamento.objects.get(pedido=pedido, pagamento=self.configuracao_pagamento.forma_pagamento)
        if valores:
            for chave in valores.keys():
                setattr(pedido_pagamento, chave, valores[chave])
        else:
            pedido_pagamento.identificador_id = self.registrador.identificador_id
        pedido_pagamento.save()
