import abc
import collections
import threading

from libtng import six
from libtng import timezone
from libtng.cqrs.runner import Runner

from exceptions import CommandAlreadyHandled


__all__ = [
    'CommandExecution',
    'GatewayHistory',
    'GatewayInterface'
]


CommandExecution = collections.namedtuple('CommandExecution', ['timestamp','command'])


class GatewayHistory(collections.Mapping, collections.Container):

    def __init__(self):
        self._history = {}
        self._lock = threading.Lock()

    def is_executed(self, command):
        """
        Return a boolean indicating if the command has been executed.
        """
        return command in self._history

    def get_command_hash(self, command):
        parts = [type(command).__module__, type(command).__name__]
        parts.append(str(hash(command)))
        return hash(''.join(parts))

    def get(self, command):
        return self._history.get(self.get_command_hash(command))

    def put(self, timestamp, command):
        with self._lock:
            self._history[self.get_command_hash(command)] = CommandExecution(timestamp, command)

    def __getitem__(self, key):
        return self._history[key]

    def __len__(self):
        return len(self._history)

    def __iter__(self):
        return iter(self._history)


class GatewayInterface(six.with_metaclass(abc.ABCMeta)):

    def __init__(self, runner=None):
        self._runner = runner or Runner()
        self._history = GatewayHistory()

    @abc.abstractmethod
    def dispatch(self, command, *args, **kwargs):
        """
        Executes the given command.
        """
        pass

    def add_to_history(self, command):
        """
        State that `command` has been succesfully executed at
        `timestamp`.

        :param command:
            the succesfully execute command.
        """
        if command not in self._history:
            self._history.put(timezone.now(), command)

    def purge_history(self):
        """
        Purges the command execution history of the :class:`GatewayInterface`.
        """
        self._history = GatewayHistory()

    def queue(self, transaction, command):
        """
        Append a `command` to the queue of a `transaction`.

        :param transaction:
            a :class:`~libtng.cqrs.Transaction` instance.
        :param command:
            an instance of a :class:`~libtng.cqrs.Command`
            implementation.
        """
        transaction.append(command)

    def rollback(self, command):
        """
        Rollback a :class:`~libtng.cqrs.Command`.
        """
        self._runner.rollback(self, command)

    def run(self, command):
        executed = self._history.get(command)
        if executed is not None:
            raise CommandAlreadyHandled(executed)
        result = self._runner.run(self, command)
        self.add_to_history(command)
        return result