import unittest

from libtng import entity


__all__ = ['PrivateTestCase']


class PrivateTestCaseEntity(entity.Entity):
    attribute    = entity.IntegerProperty(required=False, public_getter=False, public_setter=False)

    def get_test_method(self):
        getattr(self, 'attribute')

    def set_test_method(self):
        setattr(self, 'attribute', 1)


class PrivateSubclassTestCaseEntity(PrivateTestCaseEntity):

    def get_test_method(self):
        getattr(self, 'attribute')

    def set_test_method(self):
        setattr(self, 'attribute', 1)


class PrivateMethodTestCaseEntity(entity.Entity):
    attribute    = entity.IntegerProperty(required=False, public_getter=False, public_setter=False)

    @entity.privatemethod
    def private_method(self):
        getattr(self, 'attribute')

    def private_method_caller(self):
        self.private_method()


class PrivateTestCase(unittest.TestCase):

    def test_method_can_get_private_attribute(self):
        instance = PrivateTestCaseEntity()
        try:
            instance.get_test_method()
        except entity.PrivateAttribute:
            raise AssertionError("Private attribute is not accessible by instance method.")

    def test_method_can_set_private_attribute(self):
        instance = PrivateTestCaseEntity()
        try:
            instance.set_test_method()
        except entity.ReadOnlyAttribute:
            raise AssertionError("Private attribute is not accessible by instance method.")

    def test_public_cannot_get_attribute(self):
        instance = PrivateTestCaseEntity()
        self.assertRaises(entity.PrivateAttribute, getattr, instance, 'attribute')

    def test_public_cannot_set_attribute(self):
        instance = PrivateTestCaseEntity()
        self.assertRaises(entity.ReadOnlyAttribute, setattr, instance, 'attribute', 1)


class PrivateSubclassTestCase(unittest.TestCase):

    def test_method_can_get_private_attribute(self):
        instance = PrivateSubclassTestCaseEntity()
        try:
            instance.get_test_method()
        except entity.PrivateAttribute:
            raise AssertionError("Private attribute is not accessible by instance method.")

    def test_method_can_set_private_attribute(self):
        instance = PrivateSubclassTestCaseEntity()
        try:
            instance.set_test_method()
        except entity.ReadOnlyAttribute:
            raise AssertionError("Private attribute is not accessible by instance method.")

    def test_public_cannot_get_attribute(self):
        instance = PrivateSubclassTestCaseEntity()
        self.assertRaises(entity.PrivateAttribute, getattr, instance, 'attribute')

    def test_public_cannot_set_attribute(self):
        instance = PrivateSubclassTestCaseEntity()
        self.assertRaises(entity.ReadOnlyAttribute, setattr, instance, 'attribute', 1)


class PrivateMethodTestCase(unittest.TestCase):

    def test_private_raises_privatemethod(self):
        instance = PrivateMethodTestCaseEntity(attribute=1)
        self.assertRaises(entity.PrivateMethod, instance.private_method)

    def test_method_can_call_private_method(self):
        instance = PrivateMethodTestCaseEntity(attribute=1)
        instance.private_method_caller()

if __name__ == '__main__':
    unittest.main()