import copy
import operator

from libtng.exceptions import ProgrammingError
from libtng import environ

from _ansible import Runner, PlayBook


MODULE      = 'module'
PLAYBOOK    = 'playbook'


class ProvisioningOperation(object):
    """
    Represents an operation to be executed by the
    :class:`~libtng.provisioning.Provisioner`;
    either an Ansible playbook or module.
    """

    def __init__(self, playbook_name, extra_params, module_name, module_args):
        if not operator.xor(*map(bool,[playbook_name,module_name])):
            raise ProgrammingError(
                "Provisionable must specify either a module or a playbook.")
        self._module_name = module_name
        self._module_args = module_args
        self._playbook_name = playbook_name
        self._extra_params = extra_params or {}

    def __eq__(self, other):
        return hash(self) == hash(other)

    def __hash__(self):
        return hash((self._module_name, self._module_args,
            self._playbook_name))

    def get_runner_class(self):
        return Runner if self._module_name else PlayBook

    def get_runner_params(self, **extra_params):
        params = {
            'base_dir': environ.getenv('ANSIBLE_PLAYBOOK_BASEDIR')
        }
        extra_params.update(self._extra_params)
        if self._module_name:
            params['module_name'] = self._module_name
            if self._module_args:
                params['module_args'] = self._module_args
        elif self._playbook_name:
            params['playbook'] = self._playbook_name
            params['extra_params'] = extra_params
        return params

    def run(self, inventory, provisionable, options, extra_params=None, result_class=lambda x: x):
        """
        Run the specified operations.

        :param inventory:
            a :class:`~libtng.provisioning.Inventory` instance
            specifying the target hosts.
        :param provisionable:
            the :class:`~libtng.provisioning.Provisionable` instance
            that invoked the operation(s).
        :param options:
            a :class:`~libtng.provisioning.ProvisionerOptions` instance
            specifying the connection options.
        """
        extra_params = extra_params or {}
        runner = options.configure_runner(self.get_runner_class(),
            inventory=inventory,
            **self.get_runner_params(**extra_params))
        return map(result_class, runner.run())