import types

from werkzeug.wsgi import responder

from libtng.http.base import HttpResponseBase
from libtng.http import HttpResponse
from libtng.exceptions.http.base import HttpException
from libtng.wsgi.request import WSGIRequest
from libtng.module_loading import import_by_path
from libtng.exceptions.http import NotFound


class WSGIApplication(object):
    """
    Implements the WSGI protocol.

    :ivar default_mimetype:
        the default mimetype for responses returned to the downstream
            client.
    :ivar request_class:
        an Python class implementing the :class:`libtng.interfaces.Request`
        interface used to represent incoming requests.
    :ivar exception_map:
        a mapping of :exc:`Exception` classes to :class:`libtng.exceptions.http`
        classes. Example usage is when an required infrastructure component
        is not reachable to return a 503.
    """
    request_class = WSGIRequest

    def __init__(self, settings, urls, views, middleware_classes=None,
        default_mimetype="text/html", request_class=None):
        self.settings = settings
        self.urls = urls
        self.views = views
        self.middleware_classes = middleware_classes or []
        self._view_middleware = []
        self._template_response_middleware = []
        self._response_middleware = []
        self._exception_middleware = []
        self._request_middleware = []
        self.default_mimetype = default_mimetype
        self.request_class = request_class or WSGIRequest
        self.exception_map = {}
        self.load_middleware()

    def load_middleware(self):
        """
        Populate middleware lists from :attr:`WSGIApplication.middleware_classes`.
        """
        self._view_middleware = []
        self._template_response_middleware = []
        self._response_middleware = []
        self._exception_middleware = []

        request_middleware = []
        for middleware_path in self.middleware_classes:
            mw_class = import_by_path(middleware_path)
            try:
                mw_instance = mw_class()
            except MiddlewareNotUsed:
                continue

            if hasattr(mw_instance, 'process_request'):
                request_middleware.append(mw_instance.process_request)
            if hasattr(mw_instance, 'process_view'):
                self._view_middleware.append(mw_instance.process_view)
            if hasattr(mw_instance, 'process_template_response'):
                self._template_response_middleware.insert(0, mw_instance.process_template_response)
            if hasattr(mw_instance, 'process_response'):
                self._response_middleware.insert(0, mw_instance.process_response)
            if hasattr(mw_instance, 'process_exception'):
                self._exception_middleware.insert(0, mw_instance.process_exception)

        # We only assign to this when initialization is complete as it is used
        # as a flag for initialization being complete.
        self._request_middleware = request_middleware

    @responder
    def wsgi_app(self, environ, start_response):
        request = self.request_class(environ)
        request.META = {} # TODO: Quick fix for Django compat.
        urls = self.urls.bind_to_environ(environ)
        try:
            response = self.process_request(request)
            if response is None:
                response = self.dispatch(urls, request)
        except HttpException as e:
            response = e.get_response()
        return response

    def dispatch(self, urls, request):
        """Dispatches a request to it's controller (view).

        Args:
            request: a :class:`libtng.wsgi.request.WSGIRequest`
                instance representing the incoming request.

        Returns:
            HttpResponse: a :class:`libtng.http.HttpResponse`
                instance representing the response data that
                is returned to the client.
        """
        response = urls.dispatch(lambda e, v: self.process_view(e, request, v),
            catch_http_exceptions=True)
        if response.__class__.__name__ == 'NotFound':
            raise NotFound(mimetype=self.default_mimetype)
        self.process_response(request, response) # Apply regardless of the response
        return response

    def process_view(self, endpoint, request, view_params):
        """Process a `request`.

        Args:
            endpoint: a string specifying the callback endpoint.
            request: a :class:`libtng.wsgi.request.WSGIRequest`
                instance representing the incoming request.
            view_params: a dictionary containing the parameters
                parsed from the view url.

        Returns:
            HttpResponse: a :class:`libtng.http.HttpResponse`
                instance representing the response data that
                is returned to the client.
            None: the middleware did not veto and the incoming
                request is processed further.
        """
        response = None
        try:
            callback = self.views[endpoint]
        except KeyError:
            response = NotFound(mimetype=self.default_mimetype).get_response()
        else:
            for middleware_method in self._view_middleware:
                response = middleware_method(request, callback, None, view_params)
                if response:
                    break
            if response is None:
                try:
                    response = callback(request, **view_params)
                except HttpException as e: # Render the HttpException as the response.
                    response = e.get_response()
                except Exception as e:
                    if not self.is_mapped_exception(e):
                        raise
                    response = self.get_exception_response(request, e)
                if response is None:
                    if isinstance(callback, types.FunctionType):    # FBV
                        view_name = callback.__name__
                    else:                                           # CBV
                        view_name = callback.__class__.__name__ + '.__call__'
                    raise ValueError("The view {0}.{1} didn't return an HttpResponse object."\
                        .format(callback.__module__, view_name))
        return response

    def process_request(self, request):
        """Apply the request middleware. This processes an
        incoming request and may set various attributes on it.

        Args:
            request: a :class:`libtng.wsgi.request.WSGIRequest`
                instance representing the incoming request.

        Returns:
            None: the middleware did not veto and the incoming
                request is processed further.
            HttpResponse: a middleware class vetoed the request-response
                cycle and the return value of :meth:`process_request`
                will be returned to the client.
        """
        response = None
        for middleware_method in self._request_middleware:
            response = middleware_method(request)
            if response:
                break
        return response

    def process_response(self, request, response):
        """Process the response just before returning it to the client.

        Args:
            request: a :class:`libtng.wsgi.request.WSGIRequest`
                instance representing the incoming request.
            response: a :class:`libtng.http.HttpResponse` instance
                representing the outgoing response.

        Returns:
            HttpResponse: the `response` object.
        """
        for middleware_method in self._response_middleware:
            response = middleware_method(request, response)

        # If the response supports deferred rendering, apply template
        # response middleware and then render the response
        if hasattr(response, 'render') and callable(response.render):
            for middleware_method in self._template_response_middleware:
                response = middleware_method(request, response)
            response = response.render()
        return response

    def __call__(self, environ, start_response):
        return self.wsgi_app(environ, start_response)

    def map_exception(self, exc_type, http_exc_type, message=None,
        template_name=None, mimetype=None):
        """
        Maps an :exc:`Exception` subclass `exc_type` to a
        :class:`~libtng.exceptions.http.HttpException` class.

        Args:
            exc_type: the raised exception class.
            http_exc_type: the returned HTTP exception.

        Kwargs:
            message: a descriptive error message.
            template_name: a path on the local filesystem to the template
                to render.
            mimetype: a string specifying the mimetype of the returned
                response.

        Returns:
            None
        """
        if any([message, template_name, mimetype]):
            raise NotImplementedError
        else:
            self.exception_map[exc_type] = lambda request: http_exc_type().get_response()

    def is_mapped_exception(self, exception):
        """
        Return a :class:`bool` indicating if :exc:`Exception` is mapped
        to a default HTTP response.
        """
        return type(exception) in self.exception_map

    def get_exception_response(self, request, exception):
        """
        Return the :class:`~libtng.http.HttpResponse` for the
        specified exception `exception`.
        """
        return self.exception_map[type(e)](request)