from argparse import ArgumentParser
from cStringIO import StringIO
import os
import sys
import getpass
import pwd
import getpass
try:
    import configparser
except ImportError:
    import ConfigParser as configparser

from libtng.encoding import force_str
from libtng.cli.colors import color_style

from libtng.exceptions import NOTIMPLEMENTED_SUBCLASS


class CommandError(Exception):
    """
    Exception class indicating a problem while executing a management
    command.

    If this exception is raised during the execution of a management
    command, it will be caught and turned into a nicely-printed error
    message to the appropriate output stream (i.e., stderr); as a
    result, raising this exception (with a sensible description of the
    error) is the preferred way to indicate that something has gone
    wrong in the execution of a command.

    """
    pass



def handle_default_options(options):
    """
    Include any default options that all commands should accept here
    so that ManagementUtility can handle them before searching for
    user commands.

    """
    if options.settings:
        os.environ['DJANGO_SETTINGS_MODULE'] = options.settings
    if options.pythonpath:
        sys.path.insert(0, options.pythonpath)


class Argument(object):

    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs

    def get_args(self):
        return self.args

    def get_kwargs(self):
        return self.kwargs


class BaseCommand(object):
    """
    The base class from which all management commands ultimately
    derive.

    Use this class if you want access to all of the mechanisms which
    parse the command-line arguments and work out what code to call in
    response; if you don't need to change any of that behavior,
    consider using one of the subclasses defined in this file.

    If you are interested in overriding/customizing various aspects of
    the command-parsing and -execution behavior, the normal flow works
    as follows:

    1. ``tngems-admin.py`` or ``manage.py`` loads the command class
       and calls its ``run_from_argv()`` method.

    2. The ``run_from_argv()`` method calls ``create_parser()`` to get
       an ``ArgumentParser`` for the arguments, parses them, performs
       any environment changes requested by options like
       ``pythonpath``, and then calls the ``execute()`` method,
       passing the parsed arguments.

    3. The ``execute()`` method attempts to carry out the command by
       calling the ``handle()`` method with the parsed arguments; any
       output produced by ``handle()`` will be printed to standard
       output and, if the command is intended to produce a block of
       SQL statements, will be wrapped in ``BEGIN`` and ``COMMIT``.

    4. If ``handle()`` or ``execute()`` raised any exception (e.g.
       ``CommandError``), ``run_from_argv()`` will  instead print an error
       message to ``stderr``.

    Thus, the ``handle()`` method is typically the starting point for
    subclasses; many built-in commands and command types either place
    all of their logic in ``handle()``, or perform some additional
    parsing work in ``handle()`` and then delegate from it to more
    specialized methods as needed.

    Several attributes affect behavior at various steps along the way:

    ``args``
        A string listing the arguments accepted by the command,
        suitable for use in help messages; e.g., a command which takes
        a list of application names might set this to '<appname
        appname ...>'.

    ``can_import_settings``
        A boolean indicating whether the command needs to be able to
        import Django settings; if ``True``, ``execute()`` will verify
        that this is possible before proceeding. Default value is
        ``True``.

    ``help``
        A short description of the command, which will be printed in
        help messages.

    ``default_option_list``
        This is the list of ``optparse`` options which will be fed
        into the command's ``ArgumentParser`` for parsing arguments.

    ``output_transaction``
        A boolean indicating whether the command outputs SQL
        statements; if ``True``, the output will automatically be
        wrapped with ``BEGIN;`` and ``COMMIT;``. Default value is
        ``False``.

    ``requires_model_validation``
        A boolean; if ``True``, validation of installed models will be
        performed prior to executing the command. Default value is
        ``True``. To validate an individual application's models
        rather than all applications' models, call
        ``self.validate(app)`` from ``handle()``, where ``app`` is the
        application's Python module.

    ``leave_locale_alone``
        A boolean indicating whether the locale set in settings should be
        preserved during the execution of the command instead of being
        forcibly set to 'en-us'.

        Default value is ``False``.

        Make sure you know what you are doing if you decide to change the value
        of this option in your custom command if it creates database content
        that is locale-sensitive and such content shouldn't contain any
        translations (like it happens e.g. with django.contrim.auth
        permissions) as making the locale differ from the de facto default
        'en-us' might cause unintended effects.

        This option can't be False when the can_import_settings option is set
        to False too because attempting to set the locale needs access to
        settings. This condition will generate a CommandError.

    ``requires_superuser``
        A boolean; indicating if the command *must* be run as root.

    ``requires_credentials``
        A boolean; indicating that executing this command required
        user credentials.

    ``system_user``
        A string; specifies the system user this command should execute
        as.
    """
    # Metadata about this command.
    default_option_list = (
        Argument('-v', '--verbosity', action='store', dest='verbosity', default='1',
            type=int, choices=[0,1,2,3,4],
            help='verbosity level; 0=minimal output, 1=normal output, 2=verbose output, 3=very verbose output'),
        Argument('--settings',
            help='the Python path to a settings module, e.g. "myproject.settings.main". If this isn\'t provided, the DJANGO_SETTINGS_MODULE environment variable will be used.'),
        Argument('--pythonpath',
            help='a directory to add to the Python path, e.g. "/home/djangoprojects/myproject".'),
        Argument('--traceback', action='store_true',
            help='raise on exception'),
        Argument('-u','--username', dest='invoking_username', default=None,
            help="specifies a system user to execute the command as.")
    )
    option_list = []
    help = ''
    args = ''
    requires_superuser = False
    requires_credentials = False
    system_user = None
    with_session_authorization = False

    # Configuration shortcuts that alter various logic.
    can_import_settings = True
    requires_model_validation = True
    output_transaction = False  # Whether to wrap the output in a "BEGIN; COMMIT;"
    leave_locale_alone = False

    def __init__(self):
        self.style = color_style()

    def get_version(self):
        """
        Return the Django version, which should be correct for all
        built-in Django commands. User-supplied commands should
        override this method.

        """
        return tngems.get_version()

    def usage(self, subcommand):
        """
        Return a brief description of how to use this command, by
        default from the attribute ``self.help``.

        """
        usage = '%(prog)s {0} [options] {1}'.format(subcommand, self.args)
        if self.help:
            return '{0}\n\n{1}'.format(usage, self.help)
        else:
            return usage

    def create_parser(self, prog_name, subcommand):
        """
        Create and return the ``ArgumentParser`` which will be used to
        parse the arguments to this command.

        """
        parser = ArgumentParser(prog=prog_name,usage=self.usage(subcommand))
        for arg in self.default_option_list:
            parser.add_argument(*arg.get_args(), **arg.get_kwargs())
        for arg in self.option_list:
            parser.add_argument(*arg.get_args(), **arg.get_kwargs())
        return parser

    def print_help(self, prog_name, subcommand):
        """
        Print the help message for this command, derived from
        ``self.usage()``.

        """
        parser = self.create_parser(prog_name, subcommand)
        parser.print_help()

    def run_from_argv(self, argv):
        """
        Set up any environment changes requested (e.g., Python path
        and Django settings), then run this command. If the
        command raises a ``CommandError``, intercept it and print it sensibly
        to stderr. If the ``--traceback`` option is present or the raised
        ``Exception`` is not ``CommandError``, raise it.
        """
        parser = self.create_parser(argv[0], argv[1])
        args = parser.parse_args(argv[2:])
        handle_default_options(args)
        try:
            self.execute(args)
        except Exception as e:
            if args.traceback or not isinstance(e, CommandError):
                raise

            # self.stderr is not guaranteed to be set here
            stderr = getattr(self, 'stderr', OutputWrapper(sys.stderr, self.style.ERROR))
            stderr.write('%s: %s' % (e.__class__.__name__, e))
            sys.exit(1)

    def validate_user_credentials(self, args):
        username = args.invoking_username
        conf = configparser.ConfigParser()
        config_src = os.path.expanduser(const.LOCAL_USER_CONF)
        try:
            conf.read([config_src])
            username = conf.get('ui','username')
            encrypted_password = conf.get('ui', 'encrypted_password')
        except (configparser.NoSectionError, configparser.NoOptionError):
            if username is None:
                print >> sys.stdout, (
                    "No local user was specified. Please specify the u"
                    "ser to execute this command as by providing the -"
                    "u parameter"
                )
                sys.exit(1)
            encrypted_password = None
        try:
            user = SystemAgent.objects.filter_by(username=username).one()
        except SystemAgent.DoesNotExist:
            print >> sys.stdout, "No such user: {0}".format(username)
            sys.exit(1)
        if encrypted_password != user.password:
            password = keyring.get_password(
                const.KEYRING_SERVICE_NAME, username)
            while password is None:
                password = getpass.getpass("Password: ")
            if not user.check_password(password):
                print >> sys.stdout, "Wrong password."
                sys.exit(1)
            if username != const.ROOT_USER:
                keyring.set_password(
                    const.KEYRING_SERVICE_NAME, username, password)
        self.user = user

        # Update config
        if user.username != const.ROOT_USER:
            if not conf.has_section('ui'):
                conf.add_section('ui')
            conf.set('ui','username', user.username)
            conf.set('ui', 'encrypted_password', user.password)
            with open(config_src, 'wb') as f:
                conf.write(f)

    def set_system_user(self):
        p = pwd.getpwnam(self.system_user)
        os.setuid(p.pw_uid)

    def execute(self, args, **options):
        """
        Try to execute this command, performing model validation if
        needed (as controlled by the attribute
        ``self.requires_model_validation``, except if force-skipped).
        """
        if self.system_user:
            self.set_system_user()
        if self.requires_superuser and (os.getuid() != 0):
            print >> sys.stderr, "Command must be run a superuser."
            sys.exit(1)
        if self.requires_credentials or self.with_session_authorization:
            self.validate_user_credentials(args)
            assert hasattr(self,'user')
            assert isinstance(self.user, SystemAgent)
        self.stdout = OutputWrapper(options.get('stdout', sys.stdout))
        self.stderr = OutputWrapper(options.get('stderr', sys.stderr), self.style.ERROR)

        saved_locale = None
        if not self.leave_locale_alone:
            # Only mess with locales if we can assume we have a working
            # settings file, because django.utils.translation requires settings
            # (The final saying about whether the i18n machinery is active will be
            # found in the value of the USE_I18N setting)
            if not self.can_import_settings:
                raise CommandError("Incompatible values of 'leave_locale_alone' "
                                   "(%s) and 'can_import_settings' (%s) command "
                                   "options." % (self.leave_locale_alone,
                                                 self.can_import_settings))
            # Switch to US English, because django-admin.py creates database
            # content like permissions, and those shouldn't contain any
            # translations.
            from libtng import translation
            saved_locale = translation.get_language()
            translation.activate('en-us')

        try:
            output = self.handle(args, **options)
        finally:
            if saved_locale is not None:
                translation.activate(saved_locale)

    def validate(self, app=None, display_num_errors=False):
        """
        Validates the given app, raising CommandError for any errors.

        If app is None, then this will validate all installed apps.

        """
        from django.core.management.validation import get_validation_errors
        s = StringIO()
        num_errors = get_validation_errors(s, app)
        if num_errors:
            s.seek(0)
            error_text = s.read()
            raise CommandError("One or more models did not validate:\n%s" % error_text)
        if display_num_errors:
            self.stdout.write("%s error%s found" % (num_errors, '' if num_errors == 1 else 's'))

    def handle(self, args, **options):
        """
        The actual logic of the command. Subclasses must implement
        this method.

        """
        raise NotImplementedError()


class NoArgsCommand(BaseCommand):
    """
    A command which takes no arguments on the command line.

    Rather than implementing ``handle()``, subclasses must implement
    ``handle_noargs()``; ``handle()`` itself is overridden to ensure
    no arguments are passed to the command.

    Attempting to pass arguments will raise ``CommandError``.

    """
    args = ''

    def handle(self, *args, **options):
        if args:
            raise CommandError("Command doesn't accept any arguments")
        return self.handle_noargs(**options)

    def handle_noargs(self, **options):
        """
        Perform this command's actions.

        """
        raise NotImplementedError()


class OutputWrapper(object):
    """
    Wrapper around stdout/stderr
    """
    def __init__(self, out, style_func=None, ending='\n'):
        self._out = out
        self.style_func = None
        if hasattr(out, 'isatty') and out.isatty():
            self.style_func = style_func
        self.ending = ending

    def __getattr__(self, name):
        return getattr(self._out, name)

    def write(self, msg, style_func=None, ending=None):
        ending = self.ending if ending is None else ending
        if ending and not msg.endswith(ending):
            msg += ending
        style_func = [f for f in (style_func, self.style_func, lambda x:x)
                      if f is not None][0]
        self._out.write(force_str(style_func(msg)))
