from libtng.exceptions import ImproperlyConfigured
from libtng.cqrs.provider import registry

__all__ = ['Runner']


class Runner(object):
    """
    Responsible for running commands. The default implementation
    only call the :meth:`libtng.cqrs.CommandHandler.handle` method
    on a command handler; override the :meth:`Runner.run` method
    for more fine-grained behavior, such as dependency injection,
    security checks, transaction management, logging, etc.
    """

    def __init__(self, handlers=None):
        """
        Initialize a new :class:`Runner` instance.

        :param handlers:
            a :class:`~libtng.cqrs.HandlersProvider` instance, or an
            object exposing a similar interface.
        """
        self._handlers = handlers or registry

    def run(self, gateway, command):
        """
        Run the specified `command`.

        :param gateway:
            an instance of the :class:`~libtng.cqrs.GatewayInterface`
            implementation that is invoking the command.
        :param command:
            a :class:`libtng.cqrs.Command` implementation.
        """
        handler = self._handlers.get_handler(gateway, self, command)
        return handler.run_command(command, handler.handle)

    def rollback(self, gateway, command):
        """
        Rollbacks a `command`.

        :param gateway:
            an instance of the :class:`~libtng.cqrs.GatewayInterface`
            implementation that is invoking the command.
        :param command:
            a :class:`libtng.cqrs.Command` implementation.
        """
        handler = self._handlers.get_handler(gateway, self, command)
        try:
            handler.run_command(command, handler.rollback)
        except AttributeError as e:
            raise ImproperlyConfigured(
                "Cannot rollback {0} because it's handler did not spec"
                "ify a `rollback()` method.".format(type(command).__name__)
            )