"""
Command-Query Responsibility Segregation framework (CQRS). Separates
destructive operations from non-destructive (read) operations; CQRS
is "the creation of two objects where there was previously one.
The separation occurs based upon whether the methods are a command
or a query." (Greg Young).

Example commands:
- Update the state of a relation in a relational database.
- Create a directory on a filesystem.

Example queries:
- Get a result set for a query on a relational database.
- Read the contents of a directory on a filesystem.
"""
import functools

from libtng.entity import inject_state

from command import *
from handler import *
from runner import *
from gateway import *
from provider import *
from transaction import *
from exceptions import *


__all__ = [
    'Command',
    'CommandHandler',
    'Runner',
    'SimpleGateway',
    'HandlersProvider',
    'Transaction',
    'MissingCommandHandler',
    'inject_state',
    'registry'
]


def handles(command_cls, provider=None):
    """
    Decorator to be used on :class:`~libtng.cqrs.CommandHandler`
    subclasses in order to specify the command they handle.

    :param command_cls:
        a :class:`~libtng.cqrs.Command` subclass.
    :param provider:
        specifies the registry where the mapping of the command
        to it's handler will be stored.
    """
    global registry
    provider = provider or registry
    def wrapper(cls):
        provider.register_handler(command_cls, cls)
        return cls

    return wrapper