import abc
import collections
import uuid

from libtng import entity
from libtng import six
from libtng.cqrs.provider import registry

__all__ = ['Command','CommandState']



PENDING     = 0x0000000000000001
IN_PROGRESS = 0x0000000000000010
FINISHED    = 0x0000000000000100
FAILED      = 0x0000000000001000
STALE       = 0x0000000000010000


class CommandState(object):
    """
    Keeps track of the state of :class:`Command`.
    """

    @property
    def status(self):
        return self.__status

    def __init__(self, command):
        self.__command = command
        self.__status = PENDING
        self.__flags = {}

    def set_flag(self, name, value):
        self.__flags[name] = value

    def get_flag(self, name):
        return self.__flags[name]

    def set_status(self, status):
        self.__status = status


class CommandMeta(abc.ABCMeta):
    pass


class Command(six.with_metaclass(CommandMeta)):
    """
    Describes a destructive command on a certain part of a system.

    A command can specify ``__unique__=True``, to
    indicate that the command may executed only once (until
    :meth:`~libtng.cqrs.gateway.GatewayInterface.purge_history()`
    is called).

    .. note::

        If you specify ``__unique__=True``, you must override the
        ``get_identity()`` method in order to allow the runner to
        establish the identity of the command.
    """
    __unique__  = False
    PENDING     = PENDING
    IN_PROGRESS = IN_PROGRESS
    FINISHED    = FINISHED
    FAILED      = FAILED
    STALE       = STALE

    def __new__(cls, *args, **kwargs):
        instance = super(Command, cls).__new__(cls, *args, **kwargs)
        instance._uuid = hash(uuid.uuid4().hex)
        return instance

    @classmethod
    def handler(cls, handler_class):
        """
        Class-decorator to use on a :class:`~libtng.cqrs.CommandHandler`
        implementation to register it for `cls`.
        """
        registry.register_handler(cls, handler_class)

    def get_identity(self):
        return self._uuid

    def __hash__(self):
        return self.get_identity()

    def is_unique(self):
        return self.__unique__

    class Meta:
        abstract = True